#include <algorithm>
#include <array>
#include <cassert>
#include <cstddef>
#include <cstdint>
#include <exception>
#include <initializer_list>
#include <iterator>
#include <new>
#include <stdexcept>
#include <string>
#include <type_traits>
#include <utility>
#if __cplusplus >= 201703L
#include <string_view>
#endif
#if __cplusplus >= 202002L
#include <ranges>
#endif

namespace rust {
inline namespace cxxbridge1 {
// #include "rust/cxx.h"

#ifndef CXXBRIDGE1_PANIC
#define CXXBRIDGE1_PANIC
template <typename Exception>
void panic [[noreturn]] (const char *msg);
#endif // CXXBRIDGE1_PANIC

struct unsafe_bitcopy_t;

namespace {
template <typename T>
class impl;
} // namespace

template <typename T>
::std::size_t size_of();
template <typename T>
::std::size_t align_of();

#ifndef CXXBRIDGE1_RUST_STRING
#define CXXBRIDGE1_RUST_STRING
class String final {
public:
  String() noexcept;
  String(const String &) noexcept;
  String(String &&) noexcept;
  ~String() noexcept;

  String(const std::string &);
  String(const char *);
  String(const char *, std::size_t);
  String(const char16_t *);
  String(const char16_t *, std::size_t);
#ifdef __cpp_char8_t
  String(const char8_t *s);
  String(const char8_t *s, std::size_t len);
#endif

  static String lossy(const std::string &) noexcept;
  static String lossy(const char *) noexcept;
  static String lossy(const char *, std::size_t) noexcept;
  static String lossy(const char16_t *) noexcept;
  static String lossy(const char16_t *, std::size_t) noexcept;

  String &operator=(const String &) & noexcept;
  String &operator=(String &&) & noexcept;

  explicit operator std::string() const;

  const char *data() const noexcept;
  std::size_t size() const noexcept;
  std::size_t length() const noexcept;
  bool empty() const noexcept;

  const char *c_str() noexcept;

  std::size_t capacity() const noexcept;
  void reserve(size_t new_cap) noexcept;

  using iterator = char *;
  iterator begin() noexcept;
  iterator end() noexcept;

  using const_iterator = const char *;
  const_iterator begin() const noexcept;
  const_iterator end() const noexcept;
  const_iterator cbegin() const noexcept;
  const_iterator cend() const noexcept;

  bool operator==(const String &) const noexcept;
  bool operator!=(const String &) const noexcept;
  bool operator<(const String &) const noexcept;
  bool operator<=(const String &) const noexcept;
  bool operator>(const String &) const noexcept;
  bool operator>=(const String &) const noexcept;

  void swap(String &) noexcept;

  String(unsafe_bitcopy_t, const String &) noexcept;

private:
  struct lossy_t;
  String(lossy_t, const char *, std::size_t) noexcept;
  String(lossy_t, const char16_t *, std::size_t) noexcept;
  friend void swap(String &lhs, String &rhs) noexcept { lhs.swap(rhs); }

  std::array<std::uintptr_t, 3> repr;
};
#endif // CXXBRIDGE1_RUST_STRING

#ifndef CXXBRIDGE1_RUST_STR
#define CXXBRIDGE1_RUST_STR
class Str final {
public:
  Str() noexcept;
  Str(const String &) noexcept;
  Str(const std::string &);
  Str(const char *);
  Str(const char *, std::size_t);

  Str &operator=(const Str &) & noexcept = default;

  explicit operator std::string() const;
#if __cplusplus >= 201703L
  explicit operator std::string_view() const;
#endif

  const char *data() const noexcept;
  std::size_t size() const noexcept;
  std::size_t length() const noexcept;
  bool empty() const noexcept;

  Str(const Str &) noexcept = default;
  ~Str() noexcept = default;

  using iterator = const char *;
  using const_iterator = const char *;
  const_iterator begin() const noexcept;
  const_iterator end() const noexcept;
  const_iterator cbegin() const noexcept;
  const_iterator cend() const noexcept;

  bool operator==(const Str &) const noexcept;
  bool operator!=(const Str &) const noexcept;
  bool operator<(const Str &) const noexcept;
  bool operator<=(const Str &) const noexcept;
  bool operator>(const Str &) const noexcept;
  bool operator>=(const Str &) const noexcept;

  void swap(Str &) noexcept;

private:
  class uninit;
  Str(uninit) noexcept;
  friend impl<Str>;

  std::array<std::uintptr_t, 2> repr;
};
#endif // CXXBRIDGE1_RUST_STR

#ifndef CXXBRIDGE1_RUST_SLICE
#define CXXBRIDGE1_RUST_SLICE
namespace detail {
template <bool>
struct copy_assignable_if {};

template <>
struct copy_assignable_if<false> {
  copy_assignable_if() noexcept = default;
  copy_assignable_if(const copy_assignable_if &) noexcept = default;
  copy_assignable_if &operator=(const copy_assignable_if &) & noexcept = delete;
  copy_assignable_if &operator=(copy_assignable_if &&) & noexcept = default;
};
} // namespace detail

template <typename T>
class Slice final
    : private detail::copy_assignable_if<std::is_const<T>::value> {
public:
  using value_type = T;

  Slice() noexcept;
  Slice(T *, std::size_t count) noexcept;

  template <typename C>
  explicit Slice(C &c) : Slice(c.data(), c.size()) {}

  Slice &operator=(const Slice<T> &) & noexcept = default;
  Slice &operator=(Slice<T> &&) & noexcept = default;

  T *data() const noexcept;
  std::size_t size() const noexcept;
  std::size_t length() const noexcept;
  bool empty() const noexcept;

  T &operator[](std::size_t n) const noexcept;
  T &at(std::size_t n) const;
  T &front() const noexcept;
  T &back() const noexcept;

  Slice(const Slice<T> &) noexcept = default;
  ~Slice() noexcept = default;

  class iterator;
  iterator begin() const noexcept;
  iterator end() const noexcept;

  void swap(Slice &) noexcept;

private:
  class uninit;
  Slice(uninit) noexcept;
  friend impl<Slice>;
  friend void sliceInit(void *, const void *, std::size_t) noexcept;
  friend void *slicePtr(const void *) noexcept;
  friend std::size_t sliceLen(const void *) noexcept;

  std::array<std::uintptr_t, 2> repr;
};

#ifdef __cpp_deduction_guides
template <typename C>
explicit Slice(C &c)
    -> Slice<std::remove_reference_t<decltype(*std::declval<C>().data())>>;
#endif // __cpp_deduction_guides

template <typename T>
class Slice<T>::iterator final {
public:
#if __cplusplus >= 202002L
  using iterator_category = std::contiguous_iterator_tag;
#else
  using iterator_category = std::random_access_iterator_tag;
#endif
  using value_type = T;
  using difference_type = std::ptrdiff_t;
  using pointer = typename std::add_pointer<T>::type;
  using reference = typename std::add_lvalue_reference<T>::type;

  reference operator*() const noexcept;
  pointer operator->() const noexcept;
  reference operator[](difference_type) const noexcept;

  iterator &operator++() noexcept;
  iterator operator++(int) noexcept;
  iterator &operator--() noexcept;
  iterator operator--(int) noexcept;

  iterator &operator+=(difference_type) noexcept;
  iterator &operator-=(difference_type) noexcept;
  iterator operator+(difference_type) const noexcept;
  friend inline iterator operator+(difference_type lhs, iterator rhs) noexcept {
    return rhs + lhs;
  }
  iterator operator-(difference_type) const noexcept;
  difference_type operator-(const iterator &) const noexcept;

  bool operator==(const iterator &) const noexcept;
  bool operator!=(const iterator &) const noexcept;
  bool operator<(const iterator &) const noexcept;
  bool operator<=(const iterator &) const noexcept;
  bool operator>(const iterator &) const noexcept;
  bool operator>=(const iterator &) const noexcept;

private:
  friend class Slice;
  void *pos;
  std::size_t stride;
};

#if __cplusplus >= 202002L
static_assert(std::ranges::contiguous_range<rust::Slice<const uint8_t>>);
static_assert(std::contiguous_iterator<rust::Slice<const uint8_t>::iterator>);
#endif

template <typename T>
Slice<T>::Slice() noexcept {
  sliceInit(this, reinterpret_cast<void *>(align_of<T>()), 0);
}

template <typename T>
Slice<T>::Slice(T *s, std::size_t count) noexcept {
  assert(s != nullptr || count == 0);
  sliceInit(this,
            s == nullptr && count == 0
                ? reinterpret_cast<void *>(align_of<T>())
                : const_cast<typename std::remove_const<T>::type *>(s),
            count);
}

template <typename T>
T *Slice<T>::data() const noexcept {
  return reinterpret_cast<T *>(slicePtr(this));
}

template <typename T>
std::size_t Slice<T>::size() const noexcept {
  return sliceLen(this);
}

template <typename T>
std::size_t Slice<T>::length() const noexcept {
  return this->size();
}

template <typename T>
bool Slice<T>::empty() const noexcept {
  return this->size() == 0;
}

template <typename T>
T &Slice<T>::operator[](std::size_t n) const noexcept {
  assert(n < this->size());
  auto ptr = static_cast<char *>(slicePtr(this)) + size_of<T>() * n;
  return *reinterpret_cast<T *>(ptr);
}

template <typename T>
T &Slice<T>::at(std::size_t n) const {
  if (n >= this->size()) {
    panic<std::out_of_range>("rust::Slice index out of range");
  }
  return (*this)[n];
}

template <typename T>
T &Slice<T>::front() const noexcept {
  assert(!this->empty());
  return (*this)[0];
}

template <typename T>
T &Slice<T>::back() const noexcept {
  assert(!this->empty());
  return (*this)[this->size() - 1];
}

template <typename T>
typename Slice<T>::iterator::reference
Slice<T>::iterator::operator*() const noexcept {
  return *static_cast<T *>(this->pos);
}

template <typename T>
typename Slice<T>::iterator::pointer
Slice<T>::iterator::operator->() const noexcept {
  return static_cast<T *>(this->pos);
}

template <typename T>
typename Slice<T>::iterator::reference Slice<T>::iterator::operator[](
    typename Slice<T>::iterator::difference_type n) const noexcept {
  auto ptr = static_cast<char *>(this->pos) + this->stride * n;
  return *reinterpret_cast<T *>(ptr);
}

template <typename T>
typename Slice<T>::iterator &Slice<T>::iterator::operator++() noexcept {
  this->pos = static_cast<char *>(this->pos) + this->stride;
  return *this;
}

template <typename T>
typename Slice<T>::iterator Slice<T>::iterator::operator++(int) noexcept {
  auto ret = iterator(*this);
  this->pos = static_cast<char *>(this->pos) + this->stride;
  return ret;
}

template <typename T>
typename Slice<T>::iterator &Slice<T>::iterator::operator--() noexcept {
  this->pos = static_cast<char *>(this->pos) - this->stride;
  return *this;
}

template <typename T>
typename Slice<T>::iterator Slice<T>::iterator::operator--(int) noexcept {
  auto ret = iterator(*this);
  this->pos = static_cast<char *>(this->pos) - this->stride;
  return ret;
}

template <typename T>
typename Slice<T>::iterator &Slice<T>::iterator::operator+=(
    typename Slice<T>::iterator::difference_type n) noexcept {
  this->pos = static_cast<char *>(this->pos) + this->stride * n;
  return *this;
}

template <typename T>
typename Slice<T>::iterator &Slice<T>::iterator::operator-=(
    typename Slice<T>::iterator::difference_type n) noexcept {
  this->pos = static_cast<char *>(this->pos) - this->stride * n;
  return *this;
}

template <typename T>
typename Slice<T>::iterator Slice<T>::iterator::operator+(
    typename Slice<T>::iterator::difference_type n) const noexcept {
  auto ret = iterator(*this);
  ret.pos = static_cast<char *>(this->pos) + this->stride * n;
  return ret;
}

template <typename T>
typename Slice<T>::iterator Slice<T>::iterator::operator-(
    typename Slice<T>::iterator::difference_type n) const noexcept {
  auto ret = iterator(*this);
  ret.pos = static_cast<char *>(this->pos) - this->stride * n;
  return ret;
}

template <typename T>
typename Slice<T>::iterator::difference_type
Slice<T>::iterator::operator-(const iterator &other) const noexcept {
  auto diff = std::distance(static_cast<char *>(other.pos),
                            static_cast<char *>(this->pos));
  return diff / static_cast<typename Slice<T>::iterator::difference_type>(
                    this->stride);
}

template <typename T>
bool Slice<T>::iterator::operator==(const iterator &other) const noexcept {
  return this->pos == other.pos;
}

template <typename T>
bool Slice<T>::iterator::operator!=(const iterator &other) const noexcept {
  return this->pos != other.pos;
}

template <typename T>
bool Slice<T>::iterator::operator<(const iterator &other) const noexcept {
  return this->pos < other.pos;
}

template <typename T>
bool Slice<T>::iterator::operator<=(const iterator &other) const noexcept {
  return this->pos <= other.pos;
}

template <typename T>
bool Slice<T>::iterator::operator>(const iterator &other) const noexcept {
  return this->pos > other.pos;
}

template <typename T>
bool Slice<T>::iterator::operator>=(const iterator &other) const noexcept {
  return this->pos >= other.pos;
}

template <typename T>
typename Slice<T>::iterator Slice<T>::begin() const noexcept {
  iterator it;
  it.pos = slicePtr(this);
  it.stride = size_of<T>();
  return it;
}

template <typename T>
typename Slice<T>::iterator Slice<T>::end() const noexcept {
  iterator it = this->begin();
  it.pos = static_cast<char *>(it.pos) + it.stride * this->size();
  return it;
}

template <typename T>
void Slice<T>::swap(Slice &rhs) noexcept {
  std::swap(*this, rhs);
}
#endif // CXXBRIDGE1_RUST_SLICE

#ifndef CXXBRIDGE1_RUST_BOX
#define CXXBRIDGE1_RUST_BOX
template <typename T>
class Box final {
public:
  using element_type = T;
  using const_pointer =
      typename std::add_pointer<typename std::add_const<T>::type>::type;
  using pointer = typename std::add_pointer<T>::type;

  Box() = delete;
  Box(Box &&) noexcept;
  ~Box() noexcept;

  explicit Box(const T &);
  explicit Box(T &&);

  Box &operator=(Box &&) & noexcept;

  const T *operator->() const noexcept;
  const T &operator*() const noexcept;
  T *operator->() noexcept;
  T &operator*() noexcept;

  template <typename... Fields>
  static Box in_place(Fields &&...);

  void swap(Box &) noexcept;

  static Box from_raw(T *) noexcept;

  T *into_raw() noexcept;

  /* Deprecated */ using value_type = element_type;

private:
  class uninit;
  class allocation;
  Box(uninit) noexcept;
  void drop() noexcept;

  friend void swap(Box &lhs, Box &rhs) noexcept { lhs.swap(rhs); }

  T *ptr;
};

template <typename T>
class Box<T>::uninit {};

template <typename T>
class Box<T>::allocation {
  static T *alloc() noexcept;
  static void dealloc(T *) noexcept;

public:
  allocation() noexcept : ptr(alloc()) {}
  ~allocation() noexcept {
    if (this->ptr) {
      dealloc(this->ptr);
    }
  }
  T *ptr;
};

template <typename T>
Box<T>::Box(Box &&other) noexcept : ptr(other.ptr) {
  other.ptr = nullptr;
}

template <typename T>
Box<T>::Box(const T &val) {
  allocation alloc;
  ::new (alloc.ptr) T(val);
  this->ptr = alloc.ptr;
  alloc.ptr = nullptr;
}

template <typename T>
Box<T>::Box(T &&val) {
  allocation alloc;
  ::new (alloc.ptr) T(std::move(val));
  this->ptr = alloc.ptr;
  alloc.ptr = nullptr;
}

template <typename T>
Box<T>::~Box() noexcept {
  if (this->ptr) {
    this->drop();
  }
}

template <typename T>
Box<T> &Box<T>::operator=(Box &&other) & noexcept {
  if (this->ptr) {
    this->drop();
  }
  this->ptr = other.ptr;
  other.ptr = nullptr;
  return *this;
}

template <typename T>
const T *Box<T>::operator->() const noexcept {
  return this->ptr;
}

template <typename T>
const T &Box<T>::operator*() const noexcept {
  return *this->ptr;
}

template <typename T>
T *Box<T>::operator->() noexcept {
  return this->ptr;
}

template <typename T>
T &Box<T>::operator*() noexcept {
  return *this->ptr;
}

template <typename T>
template <typename... Fields>
Box<T> Box<T>::in_place(Fields &&...fields) {
  allocation alloc;
  auto ptr = alloc.ptr;
  ::new (ptr) T{std::forward<Fields>(fields)...};
  alloc.ptr = nullptr;
  return from_raw(ptr);
}

template <typename T>
void Box<T>::swap(Box &rhs) noexcept {
  using std::swap;
  swap(this->ptr, rhs.ptr);
}

template <typename T>
Box<T> Box<T>::from_raw(T *raw) noexcept {
  Box box = uninit{};
  box.ptr = raw;
  return box;
}

template <typename T>
T *Box<T>::into_raw() noexcept {
  T *raw = this->ptr;
  this->ptr = nullptr;
  return raw;
}

template <typename T>
Box<T>::Box(uninit) noexcept {}
#endif // CXXBRIDGE1_RUST_BOX

#ifndef CXXBRIDGE1_RUST_BITCOPY_T
#define CXXBRIDGE1_RUST_BITCOPY_T
struct unsafe_bitcopy_t final {
  explicit unsafe_bitcopy_t() = default;
};
#endif // CXXBRIDGE1_RUST_BITCOPY_T

#ifndef CXXBRIDGE1_RUST_VEC
#define CXXBRIDGE1_RUST_VEC
template <typename T>
class Vec final {
public:
  using value_type = T;

  Vec() noexcept;
  Vec(std::initializer_list<T>);
  Vec(const Vec &);
  Vec(Vec &&) noexcept;
  ~Vec() noexcept;

  Vec &operator=(Vec &&) & noexcept;
  Vec &operator=(const Vec &) &;

  std::size_t size() const noexcept;
  bool empty() const noexcept;
  const T *data() const noexcept;
  T *data() noexcept;
  std::size_t capacity() const noexcept;

  const T &operator[](std::size_t n) const noexcept;
  const T &at(std::size_t n) const;
  const T &front() const noexcept;
  const T &back() const noexcept;

  T &operator[](std::size_t n) noexcept;
  T &at(std::size_t n);
  T &front() noexcept;
  T &back() noexcept;

  void reserve(std::size_t new_cap);
  void push_back(const T &value);
  void push_back(T &&value);
  template <typename... Args>
  void emplace_back(Args &&...args);
  void truncate(std::size_t len);
  void clear();

  using iterator = typename Slice<T>::iterator;
  iterator begin() noexcept;
  iterator end() noexcept;

  using const_iterator = typename Slice<const T>::iterator;
  const_iterator begin() const noexcept;
  const_iterator end() const noexcept;
  const_iterator cbegin() const noexcept;
  const_iterator cend() const noexcept;

  void swap(Vec &) noexcept;

  Vec(unsafe_bitcopy_t, const Vec &) noexcept;

private:
  void reserve_total(std::size_t new_cap) noexcept;
  void set_len(std::size_t len) noexcept;
  void drop() noexcept;

  friend void swap(Vec &lhs, Vec &rhs) noexcept { lhs.swap(rhs); }

  std::array<std::uintptr_t, 3> repr;
};

template <typename T>
Vec<T>::Vec(std::initializer_list<T> init) : Vec{} {
  this->reserve_total(init.size());
  std::move(init.begin(), init.end(), std::back_inserter(*this));
}

template <typename T>
Vec<T>::Vec(const Vec &other) : Vec() {
  this->reserve_total(other.size());
  std::copy(other.begin(), other.end(), std::back_inserter(*this));
}

template <typename T>
Vec<T>::Vec(Vec &&other) noexcept : repr(other.repr) {
  new (&other) Vec();
}

template <typename T>
Vec<T>::~Vec() noexcept {
  this->drop();
}

template <typename T>
Vec<T> &Vec<T>::operator=(Vec &&other) & noexcept {
  this->drop();
  this->repr = other.repr;
  new (&other) Vec();
  return *this;
}

template <typename T>
Vec<T> &Vec<T>::operator=(const Vec &other) & {
  if (this != &other) {
    this->drop();
    new (this) Vec(other);
  }
  return *this;
}

template <typename T>
bool Vec<T>::empty() const noexcept {
  return this->size() == 0;
}

template <typename T>
T *Vec<T>::data() noexcept {
  return const_cast<T *>(const_cast<const Vec<T> *>(this)->data());
}

template <typename T>
const T &Vec<T>::operator[](std::size_t n) const noexcept {
  assert(n < this->size());
  auto data = reinterpret_cast<const char *>(this->data());
  return *reinterpret_cast<const T *>(data + n * size_of<T>());
}

template <typename T>
const T &Vec<T>::at(std::size_t n) const {
  if (n >= this->size()) {
    panic<std::out_of_range>("rust::Vec index out of range");
  }
  return (*this)[n];
}

template <typename T>
const T &Vec<T>::front() const noexcept {
  assert(!this->empty());
  return (*this)[0];
}

template <typename T>
const T &Vec<T>::back() const noexcept {
  assert(!this->empty());
  return (*this)[this->size() - 1];
}

template <typename T>
T &Vec<T>::operator[](std::size_t n) noexcept {
  assert(n < this->size());
  auto data = reinterpret_cast<char *>(this->data());
  return *reinterpret_cast<T *>(data + n * size_of<T>());
}

template <typename T>
T &Vec<T>::at(std::size_t n) {
  if (n >= this->size()) {
    panic<std::out_of_range>("rust::Vec index out of range");
  }
  return (*this)[n];
}

template <typename T>
T &Vec<T>::front() noexcept {
  assert(!this->empty());
  return (*this)[0];
}

template <typename T>
T &Vec<T>::back() noexcept {
  assert(!this->empty());
  return (*this)[this->size() - 1];
}

template <typename T>
void Vec<T>::reserve(std::size_t new_cap) {
  this->reserve_total(new_cap);
}

template <typename T>
void Vec<T>::push_back(const T &value) {
  this->emplace_back(value);
}

template <typename T>
void Vec<T>::push_back(T &&value) {
  this->emplace_back(std::move(value));
}

template <typename T>
template <typename... Args>
void Vec<T>::emplace_back(Args &&...args) {
  auto size = this->size();
  this->reserve_total(size + 1);
  ::new (reinterpret_cast<T *>(reinterpret_cast<char *>(this->data()) +
                               size * size_of<T>()))
      T(std::forward<Args>(args)...);
  this->set_len(size + 1);
}

template <typename T>
void Vec<T>::clear() {
  this->truncate(0);
}

template <typename T>
typename Vec<T>::iterator Vec<T>::begin() noexcept {
  return Slice<T>(this->data(), this->size()).begin();
}

template <typename T>
typename Vec<T>::iterator Vec<T>::end() noexcept {
  return Slice<T>(this->data(), this->size()).end();
}

template <typename T>
typename Vec<T>::const_iterator Vec<T>::begin() const noexcept {
  return this->cbegin();
}

template <typename T>
typename Vec<T>::const_iterator Vec<T>::end() const noexcept {
  return this->cend();
}

template <typename T>
typename Vec<T>::const_iterator Vec<T>::cbegin() const noexcept {
  return Slice<const T>(this->data(), this->size()).begin();
}

template <typename T>
typename Vec<T>::const_iterator Vec<T>::cend() const noexcept {
  return Slice<const T>(this->data(), this->size()).end();
}

template <typename T>
void Vec<T>::swap(Vec &rhs) noexcept {
  using std::swap;
  swap(this->repr, rhs.repr);
}

template <typename T>
Vec<T>::Vec(unsafe_bitcopy_t, const Vec &bits) noexcept : repr(bits.repr) {}
#endif // CXXBRIDGE1_RUST_VEC

#ifndef CXXBRIDGE1_RUST_ERROR
#define CXXBRIDGE1_RUST_ERROR
class Error final : public std::exception {
public:
  Error(const Error &);
  Error(Error &&) noexcept;
  ~Error() noexcept override;

  Error &operator=(const Error &) &;
  Error &operator=(Error &&) & noexcept;

  const char *what() const noexcept override;

private:
  Error() noexcept = default;
  friend impl<Error>;
  const char *msg;
  std::size_t len;
};
#endif // CXXBRIDGE1_RUST_ERROR

#ifndef CXXBRIDGE1_RUST_OPAQUE
#define CXXBRIDGE1_RUST_OPAQUE
class Opaque {
public:
  Opaque() = delete;
  Opaque(const Opaque &) = delete;
  ~Opaque() = delete;
};
#endif // CXXBRIDGE1_RUST_OPAQUE

#ifndef CXXBRIDGE1_IS_COMPLETE
#define CXXBRIDGE1_IS_COMPLETE
namespace detail {
namespace {
template <typename T, typename = std::size_t>
struct is_complete : std::false_type {};
template <typename T>
struct is_complete<T, decltype(sizeof(T))> : std::true_type {};
} // namespace
} // namespace detail
#endif // CXXBRIDGE1_IS_COMPLETE

#ifndef CXXBRIDGE1_LAYOUT
#define CXXBRIDGE1_LAYOUT
class layout {
  template <typename T>
  friend std::size_t size_of();
  template <typename T>
  friend std::size_t align_of();
  template <typename T>
  static typename std::enable_if<std::is_base_of<Opaque, T>::value,
                                 std::size_t>::type
  do_size_of() {
    return T::layout::size();
  }
  template <typename T>
  static typename std::enable_if<!std::is_base_of<Opaque, T>::value,
                                 std::size_t>::type
  do_size_of() {
    return sizeof(T);
  }
  template <typename T>
  static
      typename std::enable_if<detail::is_complete<T>::value, std::size_t>::type
      size_of() {
    return do_size_of<T>();
  }
  template <typename T>
  static typename std::enable_if<std::is_base_of<Opaque, T>::value,
                                 std::size_t>::type
  do_align_of() {
    return T::layout::align();
  }
  template <typename T>
  static typename std::enable_if<!std::is_base_of<Opaque, T>::value,
                                 std::size_t>::type
  do_align_of() {
    return alignof(T);
  }
  template <typename T>
  static
      typename std::enable_if<detail::is_complete<T>::value, std::size_t>::type
      align_of() {
    return do_align_of<T>();
  }
};

template <typename T>
std::size_t size_of() {
  return layout::size_of<T>();
}

template <typename T>
std::size_t align_of() {
  return layout::align_of<T>();
}
#endif // CXXBRIDGE1_LAYOUT

class Str::uninit {};
inline Str::Str(uninit) noexcept {}

template <typename T>
class Slice<T>::uninit {};
template <typename T>
inline Slice<T>::Slice(uninit) noexcept {}

namespace repr {
using Fat = ::std::array<::std::uintptr_t, 2>;

struct PtrLen final {
  void *ptr;
  ::std::size_t len;
};
} // namespace repr

namespace detail {
template <typename T, typename = void *>
struct operator_new {
  void *operator()(::std::size_t sz) { return ::operator new(sz); }
};

template <typename T>
struct operator_new<T, decltype(T::operator new(sizeof(T)))> {
  void *operator()(::std::size_t sz) { return T::operator new(sz); }
};
} // namespace detail

template <typename T>
union ManuallyDrop {
  T value;
  ManuallyDrop(T &&value) : value(::std::move(value)) {}
  ~ManuallyDrop() {}
};

template <typename T>
union MaybeUninit {
  T value;
  void *operator new(::std::size_t sz) { return detail::operator_new<T>{}(sz); }
  MaybeUninit() {}
  ~MaybeUninit() {}
};

namespace {
template <>
class impl<Str> final {
public:
  static Str new_unchecked(repr::Fat repr) noexcept {
    Str str = Str::uninit{};
    str.repr = repr;
    return str;
  }
};

template <typename T>
class impl<Slice<T>> final {
public:
  static Slice<T> slice(repr::Fat repr) noexcept {
    Slice<T> slice = typename Slice<T>::uninit{};
    slice.repr = repr;
    return slice;
  }
};

template <>
class impl<Error> final {
public:
  static Error error(repr::PtrLen repr) noexcept {
    Error error;
    error.msg = static_cast<char const *>(repr.ptr);
    error.len = repr.len;
    return error;
  }
};
} // namespace
} // namespace cxxbridge1
} // namespace rust

#if __cplusplus >= 201402L
#define CXX_DEFAULT_VALUE(value) = value
#else
#define CXX_DEFAULT_VALUE(value)
#endif

namespace loot {
  namespace rust {
    enum class GameType : ::std::uint8_t;
    enum class MessageType : ::std::uint8_t;
    enum class TagSuggestion : ::std::uint8_t;
    enum class EdgeType : ::std::uint8_t;
    enum class LogLevel : ::std::uint8_t;
    struct OptionalMessageContentRef;
    struct Game;
    struct Database;
    struct Message;
    struct MessageContent;
    struct Group;
    struct Plugin;
    struct OptionalPlugin;
    struct OptionalCrc;
    struct Vertex;
    struct OptionalPluginMetadata;
    struct PluginMetadata;
    struct File;
    struct Filename;
    struct Tag;
    struct PluginCleaningData;
    struct Location;
  }
}

namespace loot {
namespace rust {
#ifndef CXXBRIDGE1_ENUM_loot$rust$GameType
#define CXXBRIDGE1_ENUM_loot$rust$GameType
enum class GameType : ::std::uint8_t {
  Oblivion = 0,
  Skyrim = 1,
  Fallout3 = 2,
  FalloutNV = 3,
  Fallout4 = 4,
  SkyrimSE = 5,
  Fallout4VR = 6,
  SkyrimVR = 7,
  Morrowind = 8,
  Starfield = 9,
  OpenMW = 10,
  OblivionRemastered = 11,
};
#endif // CXXBRIDGE1_ENUM_loot$rust$GameType

#ifndef CXXBRIDGE1_ENUM_loot$rust$MessageType
#define CXXBRIDGE1_ENUM_loot$rust$MessageType
enum class MessageType : ::std::uint8_t {
  Say = 0,
  Warn = 1,
  Error = 2,
};
#endif // CXXBRIDGE1_ENUM_loot$rust$MessageType

#ifndef CXXBRIDGE1_ENUM_loot$rust$TagSuggestion
#define CXXBRIDGE1_ENUM_loot$rust$TagSuggestion
enum class TagSuggestion : ::std::uint8_t {
  Addition = 0,
  Removal = 1,
};
#endif // CXXBRIDGE1_ENUM_loot$rust$TagSuggestion

#ifndef CXXBRIDGE1_ENUM_loot$rust$EdgeType
#define CXXBRIDGE1_ENUM_loot$rust$EdgeType
enum class EdgeType : ::std::uint8_t {
  // A special value that indicates that there is no edge.
  None = 0,
  Hardcoded = 1,
  MasterFlag = 2,
  Master = 3,
  MasterlistRequirement = 4,
  UserRequirement = 5,
  MasterlistLoadAfter = 6,
  UserLoadAfter = 7,
  MasterlistGroup = 8,
  UserGroup = 9,
  RecordOverlap = 10,
  AssetOverlap = 11,
  TieBreak = 12,
  BlueprintMaster = 13,
};
#endif // CXXBRIDGE1_ENUM_loot$rust$EdgeType

#ifndef CXXBRIDGE1_ENUM_loot$rust$LogLevel
#define CXXBRIDGE1_ENUM_loot$rust$LogLevel
enum class LogLevel : ::std::uint8_t {
  Trace = 0,
  Debug = 1,
  Info = 2,
  Warning = 3,
  Error = 4,
};
#endif // CXXBRIDGE1_ENUM_loot$rust$LogLevel

#ifndef CXXBRIDGE1_STRUCT_loot$rust$OptionalMessageContentRef
#define CXXBRIDGE1_STRUCT_loot$rust$OptionalMessageContentRef
struct OptionalMessageContentRef final {
  ::loot::rust::MessageContent const *pointer CXX_DEFAULT_VALUE(nullptr);

  bool is_some() const noexcept;
  ::loot::rust::MessageContent const &as_ref() const;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_loot$rust$OptionalMessageContentRef

#ifndef CXXBRIDGE1_STRUCT_loot$rust$Game
#define CXXBRIDGE1_STRUCT_loot$rust$Game
struct Game final : public ::rust::Opaque {
  ::loot::rust::GameType game_type() const;
  ::rust::Vec<::rust::String> additional_data_paths() const;
  void set_additional_data_paths(::rust::Slice<::rust::Str const> additional_data_paths);
  ::rust::Box<::loot::rust::Database> database() const noexcept;
  bool is_valid_plugin(::rust::Str plugin_path) const noexcept;
  void load_plugins(::rust::Slice<::rust::Str const> plugin_paths);
  void load_plugin_headers(::rust::Slice<::rust::Str const> plugin_paths);
  void clear_loaded_plugins() noexcept;
  ::rust::Box<::loot::rust::OptionalPlugin> plugin(::rust::Str plugin_name) const noexcept;
  ::rust::Vec<::loot::rust::Plugin> loaded_plugins() const noexcept;
  ::rust::Vec<::rust::String> sort_plugins(::rust::Slice<::rust::Str const> plugin_names) const;
  void load_current_load_order_state();
  bool is_load_order_ambiguous() const;
  ::rust::String active_plugins_file_path() const;
  bool is_plugin_active(::rust::Str plugin_name) const noexcept;
  ::rust::Vec<::rust::String> load_order() const noexcept;
  void set_load_order(::rust::Slice<::rust::Str const> load_order);
  ~Game() = delete;

private:
  friend ::rust::layout;
  struct layout {
    static ::std::size_t size() noexcept;
    static ::std::size_t align() noexcept;
  };
};
#endif // CXXBRIDGE1_STRUCT_loot$rust$Game

#ifndef CXXBRIDGE1_STRUCT_loot$rust$Database
#define CXXBRIDGE1_STRUCT_loot$rust$Database
struct Database final : public ::rust::Opaque {
  void load_masterlist(::rust::Str path) const;
  void load_masterlist_with_prelude(::rust::Str masterlist_path, ::rust::Str prelude_path) const;
  void load_userlist(::rust::Str path) const;
  void write_user_metadata(::rust::Str output_path, bool overwrite) const;
  void write_minimal_list(::rust::Str output_path, bool overwrite) const;
  bool evaluate(::rust::Str condition) const;
  ::rust::Vec<::rust::String> known_bash_tags() const;
  ::rust::Vec<::loot::rust::Message> general_messages(bool evaluate_conditions) const;
  ::rust::Vec<::loot::rust::Group> groups(bool include_user_metadata) const;
  ::rust::Vec<::loot::rust::Group> user_groups() const;
  void set_user_groups(::rust::Slice<::rust::Box<::loot::rust::Group> const> groups) const;
  ::rust::Vec<::loot::rust::Vertex> groups_path(::rust::Str from_group_name, ::rust::Str to_group_name) const;
  ::rust::Box<::loot::rust::OptionalPluginMetadata> plugin_metadata(::rust::Str plugin_name, bool include_user_metadata, bool evaluate_conditions) const;
  ::rust::Box<::loot::rust::OptionalPluginMetadata> plugin_user_metadata(::rust::Str plugin_name, bool evaluate_conditions) const;
  void set_plugin_user_metadata(::rust::Box<::loot::rust::PluginMetadata> plugin_metadata);
  void discard_plugin_user_metadata(::rust::Str plugin) const;
  void discard_all_user_metadata() const;
  ~Database() = delete;

private:
  friend ::rust::layout;
  struct layout {
    static ::std::size_t size() noexcept;
    static ::std::size_t align() noexcept;
  };
};
#endif // CXXBRIDGE1_STRUCT_loot$rust$Database

#ifndef CXXBRIDGE1_STRUCT_loot$rust$Message
#define CXXBRIDGE1_STRUCT_loot$rust$Message
struct Message final : public ::rust::Opaque {
  ::loot::rust::MessageType message_type() const noexcept;
  ::rust::Slice<::loot::rust::MessageContent const> content() const noexcept;
  ::rust::Str condition() const noexcept;
  ::rust::Box<::loot::rust::Message> boxed_clone() const noexcept;
  ~Message() = delete;

private:
  friend ::rust::layout;
  struct layout {
    static ::std::size_t size() noexcept;
    static ::std::size_t align() noexcept;
  };
};
#endif // CXXBRIDGE1_STRUCT_loot$rust$Message

#ifndef CXXBRIDGE1_STRUCT_loot$rust$MessageContent
#define CXXBRIDGE1_STRUCT_loot$rust$MessageContent
struct MessageContent final : public ::rust::Opaque {
  ::rust::Str text() const noexcept;
  ::rust::Str language() const noexcept;
  ::rust::Box<::loot::rust::MessageContent> boxed_clone() const noexcept;
  ~MessageContent() = delete;

private:
  friend ::rust::layout;
  struct layout {
    static ::std::size_t size() noexcept;
    static ::std::size_t align() noexcept;
  };
};
#endif // CXXBRIDGE1_STRUCT_loot$rust$MessageContent

#ifndef CXXBRIDGE1_STRUCT_loot$rust$Group
#define CXXBRIDGE1_STRUCT_loot$rust$Group
struct Group final : public ::rust::Opaque {
  ::rust::Box<::loot::rust::Group> boxed_clone() const noexcept;
  ::rust::Str name() const noexcept;
  ::rust::Str description() const noexcept;
  ::rust::Slice<::rust::String const> after_groups() const noexcept;
  ~Group() = delete;

private:
  friend ::rust::layout;
  struct layout {
    static ::std::size_t size() noexcept;
    static ::std::size_t align() noexcept;
  };
};
#endif // CXXBRIDGE1_STRUCT_loot$rust$Group

#ifndef CXXBRIDGE1_STRUCT_loot$rust$Plugin
#define CXXBRIDGE1_STRUCT_loot$rust$Plugin
struct Plugin final : public ::rust::Opaque {
  ::rust::Str name() const noexcept;

  // NaN is used to indicate that the header version was not found.
  float header_version() const noexcept;

  // An empty string is used to indicate that no version was found.
  ::rust::Str version() const noexcept;

  ::rust::Vec<::rust::String> masters() const;
  ::rust::Slice<::rust::String const> bash_tags() const noexcept;
  ::rust::Box<::loot::rust::OptionalCrc> crc() const noexcept;
  bool is_master() const noexcept;
  bool is_light_plugin() const noexcept;
  bool is_medium_plugin() const noexcept;
  bool is_update_plugin() const noexcept;
  bool is_blueprint_plugin() const noexcept;
  bool is_valid_as_light_plugin() const;
  bool is_valid_as_medium_plugin() const;
  bool is_valid_as_update_plugin() const;
  bool is_empty() const noexcept;
  bool loads_archive() const noexcept;
  bool do_records_overlap(::loot::rust::Plugin const &plugin) const;
  ::rust::Box<::loot::rust::Plugin> boxed_clone() const noexcept;
  ~Plugin() = delete;

private:
  friend ::rust::layout;
  struct layout {
    static ::std::size_t size() noexcept;
    static ::std::size_t align() noexcept;
  };
};
#endif // CXXBRIDGE1_STRUCT_loot$rust$Plugin

#ifndef CXXBRIDGE1_STRUCT_loot$rust$OptionalPlugin
#define CXXBRIDGE1_STRUCT_loot$rust$OptionalPlugin
struct OptionalPlugin final : public ::rust::Opaque {
  bool is_some() const noexcept;
  ::loot::rust::Plugin const &as_ref() const;
  ~OptionalPlugin() = delete;

private:
  friend ::rust::layout;
  struct layout {
    static ::std::size_t size() noexcept;
    static ::std::size_t align() noexcept;
  };
};
#endif // CXXBRIDGE1_STRUCT_loot$rust$OptionalPlugin

#ifndef CXXBRIDGE1_STRUCT_loot$rust$OptionalCrc
#define CXXBRIDGE1_STRUCT_loot$rust$OptionalCrc
struct OptionalCrc final : public ::rust::Opaque {
  bool is_some() const noexcept;
  ::std::uint32_t const &as_ref() const;
  ~OptionalCrc() = delete;

private:
  friend ::rust::layout;
  struct layout {
    static ::std::size_t size() noexcept;
    static ::std::size_t align() noexcept;
  };
};
#endif // CXXBRIDGE1_STRUCT_loot$rust$OptionalCrc

#ifndef CXXBRIDGE1_STRUCT_loot$rust$Vertex
#define CXXBRIDGE1_STRUCT_loot$rust$Vertex
struct Vertex final : public ::rust::Opaque {
  ::rust::Str name() const noexcept;
  ::loot::rust::EdgeType out_edge_type() const;
  ::rust::Box<::loot::rust::Vertex> boxed_clone() const noexcept;
  ~Vertex() = delete;

private:
  friend ::rust::layout;
  struct layout {
    static ::std::size_t size() noexcept;
    static ::std::size_t align() noexcept;
  };
};
#endif // CXXBRIDGE1_STRUCT_loot$rust$Vertex

#ifndef CXXBRIDGE1_STRUCT_loot$rust$OptionalPluginMetadata
#define CXXBRIDGE1_STRUCT_loot$rust$OptionalPluginMetadata
struct OptionalPluginMetadata final : public ::rust::Opaque {
  bool is_some() const noexcept;
  ::loot::rust::PluginMetadata const &as_ref() const;
  ~OptionalPluginMetadata() = delete;

private:
  friend ::rust::layout;
  struct layout {
    static ::std::size_t size() noexcept;
    static ::std::size_t align() noexcept;
  };
};
#endif // CXXBRIDGE1_STRUCT_loot$rust$OptionalPluginMetadata

#ifndef CXXBRIDGE1_STRUCT_loot$rust$PluginMetadata
#define CXXBRIDGE1_STRUCT_loot$rust$PluginMetadata
struct PluginMetadata final : public ::rust::Opaque {
  ::rust::Str name() const noexcept;

  // An empty string is used to indicate that no group is set.
  ::rust::Str group() const noexcept;

  ::rust::Slice<::loot::rust::File const> load_after_files() const noexcept;
  ::rust::Slice<::loot::rust::File const> requirements() const noexcept;
  ::rust::Slice<::loot::rust::File const> incompatibilities() const noexcept;
  ::rust::Slice<::loot::rust::Message const> messages() const noexcept;
  ::rust::Slice<::loot::rust::Tag const> tags() const noexcept;
  ::rust::Slice<::loot::rust::PluginCleaningData const> dirty_info() const noexcept;
  ::rust::Slice<::loot::rust::PluginCleaningData const> clean_info() const noexcept;
  ::rust::Slice<::loot::rust::Location const> locations() const noexcept;
  void set_group(::rust::String group) noexcept;
  void unset_group() noexcept;
  void set_load_after_files(::rust::Slice<::rust::Box<::loot::rust::File> const> files) noexcept;
  void set_requirements(::rust::Slice<::rust::Box<::loot::rust::File> const> files) noexcept;
  void set_incompatibilities(::rust::Slice<::rust::Box<::loot::rust::File> const> files) noexcept;
  void set_messages(::rust::Slice<::rust::Box<::loot::rust::Message> const> messages) noexcept;
  void set_tags(::rust::Slice<::rust::Box<::loot::rust::Tag> const> tags) noexcept;
  void set_dirty_info(::rust::Slice<::rust::Box<::loot::rust::PluginCleaningData> const> info) noexcept;
  void set_clean_info(::rust::Slice<::rust::Box<::loot::rust::PluginCleaningData> const> info) noexcept;
  void set_locations(::rust::Slice<::rust::Box<::loot::rust::Location> const> locations) noexcept;
  void merge_metadata(::loot::rust::PluginMetadata const &plugin) noexcept;
  bool has_name_only() const noexcept;
  bool is_regex_plugin() const noexcept;
  bool name_matches(::rust::Str other_name) const noexcept;
  ::rust::String as_yaml() const noexcept;
  ::rust::Box<::loot::rust::PluginMetadata> boxed_clone() const noexcept;
  ~PluginMetadata() = delete;

private:
  friend ::rust::layout;
  struct layout {
    static ::std::size_t size() noexcept;
    static ::std::size_t align() noexcept;
  };
};
#endif // CXXBRIDGE1_STRUCT_loot$rust$PluginMetadata

#ifndef CXXBRIDGE1_STRUCT_loot$rust$File
#define CXXBRIDGE1_STRUCT_loot$rust$File
struct File final : public ::rust::Opaque {
  ::loot::rust::Filename const &filename() const noexcept;
  ::rust::Str display_name() const noexcept;
  ::rust::Slice<::loot::rust::MessageContent const> detail() const noexcept;
  ::rust::Str condition() const noexcept;
  ::rust::Str constraint() const noexcept;
  ::rust::Box<::loot::rust::File> boxed_clone() const noexcept;
  ~File() = delete;

private:
  friend ::rust::layout;
  struct layout {
    static ::std::size_t size() noexcept;
    static ::std::size_t align() noexcept;
  };
};
#endif // CXXBRIDGE1_STRUCT_loot$rust$File

#ifndef CXXBRIDGE1_STRUCT_loot$rust$Filename
#define CXXBRIDGE1_STRUCT_loot$rust$Filename
struct Filename final : public ::rust::Opaque {
  ::rust::Str as_str() const noexcept;
  ::rust::Box<::loot::rust::Filename> boxed_clone() const noexcept;
  bool eq(::loot::rust::Filename const &other) const noexcept;
  bool ne(::loot::rust::Filename const &other) const noexcept;
  bool lt(::loot::rust::Filename const &other) const noexcept;
  bool le(::loot::rust::Filename const &other) const noexcept;
  bool gt(::loot::rust::Filename const &other) const noexcept;
  bool ge(::loot::rust::Filename const &other) const noexcept;
  ~Filename() = delete;

private:
  friend ::rust::layout;
  struct layout {
    static ::std::size_t size() noexcept;
    static ::std::size_t align() noexcept;
  };
};
#endif // CXXBRIDGE1_STRUCT_loot$rust$Filename

#ifndef CXXBRIDGE1_STRUCT_loot$rust$Tag
#define CXXBRIDGE1_STRUCT_loot$rust$Tag
struct Tag final : public ::rust::Opaque {
  ::rust::Str name() const noexcept;
  bool is_addition() const noexcept;
  ::rust::Str condition() const noexcept;
  ::rust::Box<::loot::rust::Tag> boxed_clone() const noexcept;
  ~Tag() = delete;

private:
  friend ::rust::layout;
  struct layout {
    static ::std::size_t size() noexcept;
    static ::std::size_t align() noexcept;
  };
};
#endif // CXXBRIDGE1_STRUCT_loot$rust$Tag

#ifndef CXXBRIDGE1_STRUCT_loot$rust$PluginCleaningData
#define CXXBRIDGE1_STRUCT_loot$rust$PluginCleaningData
struct PluginCleaningData final : public ::rust::Opaque {
  ::std::uint32_t crc() const noexcept;
  ::std::uint32_t itm_count() const noexcept;
  ::std::uint32_t deleted_reference_count() const noexcept;
  ::std::uint32_t deleted_navmesh_count() const noexcept;
  ::rust::Str cleaning_utility() const noexcept;
  ::rust::Slice<::loot::rust::MessageContent const> detail() const noexcept;
  ::rust::Box<::loot::rust::PluginCleaningData> boxed_clone() const noexcept;
  ~PluginCleaningData() = delete;

private:
  friend ::rust::layout;
  struct layout {
    static ::std::size_t size() noexcept;
    static ::std::size_t align() noexcept;
  };
};
#endif // CXXBRIDGE1_STRUCT_loot$rust$PluginCleaningData

#ifndef CXXBRIDGE1_STRUCT_loot$rust$Location
#define CXXBRIDGE1_STRUCT_loot$rust$Location
struct Location final : public ::rust::Opaque {
  ::rust::Str url() const noexcept;
  ::rust::Str name() const noexcept;
  ::rust::Box<::loot::rust::Location> boxed_clone() const noexcept;
  ~Location() = delete;

private:
  friend ::rust::layout;
  struct layout {
    static ::std::size_t size() noexcept;
    static ::std::size_t align() noexcept;
  };
};
#endif // CXXBRIDGE1_STRUCT_loot$rust$Location

extern "C" {
bool loot$rust$cxxbridge1$OptionalMessageContentRef$is_some(::loot::rust::OptionalMessageContentRef const &self) noexcept;

::rust::repr::PtrLen loot$rust$cxxbridge1$OptionalMessageContentRef$as_ref(::loot::rust::OptionalMessageContentRef const &self, ::loot::rust::MessageContent const **return$) noexcept;

::rust::repr::PtrLen loot$rust$cxxbridge1$set_log_level(::loot::rust::LogLevel level) noexcept;

bool loot$rust$cxxbridge1$is_compatible(::std::uint32_t major, ::std::uint32_t minor, ::std::uint32_t patch) noexcept;

void loot$rust$cxxbridge1$libloot_version(::rust::String *return$) noexcept;

void loot$rust$cxxbridge1$libloot_revision(::rust::String *return$) noexcept;

void loot$rust$cxxbridge1$select_message_content(::rust::Slice<::loot::rust::MessageContent const> contents, ::rust::Str language, ::loot::rust::OptionalMessageContentRef *return$) noexcept;
::std::size_t loot$rust$cxxbridge1$Game$operator$sizeof() noexcept;
::std::size_t loot$rust$cxxbridge1$Game$operator$alignof() noexcept;

::rust::repr::PtrLen loot$rust$cxxbridge1$new_game(::loot::rust::GameType game_type, ::rust::Str game_path, ::rust::Box<::loot::rust::Game> *return$) noexcept;

::rust::repr::PtrLen loot$rust$cxxbridge1$new_game_with_local_path(::loot::rust::GameType game_type, ::rust::Str game_path, ::rust::Str game_local_path, ::rust::Box<::loot::rust::Game> *return$) noexcept;

::rust::repr::PtrLen loot$rust$cxxbridge1$Game$game_type(::loot::rust::Game const &self, ::loot::rust::GameType *return$) noexcept;

::rust::repr::PtrLen loot$rust$cxxbridge1$Game$additional_data_paths(::loot::rust::Game const &self, ::rust::Vec<::rust::String> *return$) noexcept;

::rust::repr::PtrLen loot$rust$cxxbridge1$Game$set_additional_data_paths(::loot::rust::Game &self, ::rust::Slice<::rust::Str const> additional_data_paths) noexcept;

::loot::rust::Database *loot$rust$cxxbridge1$Game$database(::loot::rust::Game const &self) noexcept;

bool loot$rust$cxxbridge1$Game$is_valid_plugin(::loot::rust::Game const &self, ::rust::Str plugin_path) noexcept;

::rust::repr::PtrLen loot$rust$cxxbridge1$Game$load_plugins(::loot::rust::Game &self, ::rust::Slice<::rust::Str const> plugin_paths) noexcept;

::rust::repr::PtrLen loot$rust$cxxbridge1$Game$load_plugin_headers(::loot::rust::Game &self, ::rust::Slice<::rust::Str const> plugin_paths) noexcept;

void loot$rust$cxxbridge1$Game$clear_loaded_plugins(::loot::rust::Game &self) noexcept;

::loot::rust::OptionalPlugin *loot$rust$cxxbridge1$Game$plugin(::loot::rust::Game const &self, ::rust::Str plugin_name) noexcept;

void loot$rust$cxxbridge1$Game$loaded_plugins(::loot::rust::Game const &self, ::rust::Vec<::loot::rust::Plugin> *return$) noexcept;

::rust::repr::PtrLen loot$rust$cxxbridge1$Game$sort_plugins(::loot::rust::Game const &self, ::rust::Slice<::rust::Str const> plugin_names, ::rust::Vec<::rust::String> *return$) noexcept;

::rust::repr::PtrLen loot$rust$cxxbridge1$Game$load_current_load_order_state(::loot::rust::Game &self) noexcept;

::rust::repr::PtrLen loot$rust$cxxbridge1$Game$is_load_order_ambiguous(::loot::rust::Game const &self, bool *return$) noexcept;

::rust::repr::PtrLen loot$rust$cxxbridge1$Game$active_plugins_file_path(::loot::rust::Game const &self, ::rust::String *return$) noexcept;

bool loot$rust$cxxbridge1$Game$is_plugin_active(::loot::rust::Game const &self, ::rust::Str plugin_name) noexcept;

void loot$rust$cxxbridge1$Game$load_order(::loot::rust::Game const &self, ::rust::Vec<::rust::String> *return$) noexcept;

::rust::repr::PtrLen loot$rust$cxxbridge1$Game$set_load_order(::loot::rust::Game &self, ::rust::Slice<::rust::Str const> load_order) noexcept;
::std::size_t loot$rust$cxxbridge1$Database$operator$sizeof() noexcept;
::std::size_t loot$rust$cxxbridge1$Database$operator$alignof() noexcept;

::rust::repr::PtrLen loot$rust$cxxbridge1$Database$load_masterlist(::loot::rust::Database const &self, ::rust::Str path) noexcept;

::rust::repr::PtrLen loot$rust$cxxbridge1$Database$load_masterlist_with_prelude(::loot::rust::Database const &self, ::rust::Str masterlist_path, ::rust::Str prelude_path) noexcept;

::rust::repr::PtrLen loot$rust$cxxbridge1$Database$load_userlist(::loot::rust::Database const &self, ::rust::Str path) noexcept;

::rust::repr::PtrLen loot$rust$cxxbridge1$Database$write_user_metadata(::loot::rust::Database const &self, ::rust::Str output_path, bool overwrite) noexcept;

::rust::repr::PtrLen loot$rust$cxxbridge1$Database$write_minimal_list(::loot::rust::Database const &self, ::rust::Str output_path, bool overwrite) noexcept;

::rust::repr::PtrLen loot$rust$cxxbridge1$Database$evaluate(::loot::rust::Database const &self, ::rust::Str condition, bool *return$) noexcept;

::rust::repr::PtrLen loot$rust$cxxbridge1$Database$known_bash_tags(::loot::rust::Database const &self, ::rust::Vec<::rust::String> *return$) noexcept;

::rust::repr::PtrLen loot$rust$cxxbridge1$Database$general_messages(::loot::rust::Database const &self, bool evaluate_conditions, ::rust::Vec<::loot::rust::Message> *return$) noexcept;

::rust::repr::PtrLen loot$rust$cxxbridge1$Database$groups(::loot::rust::Database const &self, bool include_user_metadata, ::rust::Vec<::loot::rust::Group> *return$) noexcept;

::rust::repr::PtrLen loot$rust$cxxbridge1$Database$user_groups(::loot::rust::Database const &self, ::rust::Vec<::loot::rust::Group> *return$) noexcept;

::rust::repr::PtrLen loot$rust$cxxbridge1$Database$set_user_groups(::loot::rust::Database const &self, ::rust::Slice<::rust::Box<::loot::rust::Group> const> groups) noexcept;

::rust::repr::PtrLen loot$rust$cxxbridge1$Database$groups_path(::loot::rust::Database const &self, ::rust::Str from_group_name, ::rust::Str to_group_name, ::rust::Vec<::loot::rust::Vertex> *return$) noexcept;

::rust::repr::PtrLen loot$rust$cxxbridge1$Database$plugin_metadata(::loot::rust::Database const &self, ::rust::Str plugin_name, bool include_user_metadata, bool evaluate_conditions, ::rust::Box<::loot::rust::OptionalPluginMetadata> *return$) noexcept;

::rust::repr::PtrLen loot$rust$cxxbridge1$Database$plugin_user_metadata(::loot::rust::Database const &self, ::rust::Str plugin_name, bool evaluate_conditions, ::rust::Box<::loot::rust::OptionalPluginMetadata> *return$) noexcept;

::rust::repr::PtrLen loot$rust$cxxbridge1$Database$set_plugin_user_metadata(::loot::rust::Database &self, ::loot::rust::PluginMetadata *plugin_metadata) noexcept;

::rust::repr::PtrLen loot$rust$cxxbridge1$Database$discard_plugin_user_metadata(::loot::rust::Database const &self, ::rust::Str plugin) noexcept;

::rust::repr::PtrLen loot$rust$cxxbridge1$Database$discard_all_user_metadata(::loot::rust::Database const &self) noexcept;
::std::size_t loot$rust$cxxbridge1$Message$operator$sizeof() noexcept;
::std::size_t loot$rust$cxxbridge1$Message$operator$alignof() noexcept;

::rust::repr::PtrLen loot$rust$cxxbridge1$new_message(::loot::rust::MessageType message_type, ::rust::String *content, ::rust::Str condition, ::rust::Box<::loot::rust::Message> *return$) noexcept;

::rust::repr::PtrLen loot$rust$cxxbridge1$multilingual_message(::loot::rust::MessageType message_type, ::rust::Slice<::rust::Box<::loot::rust::MessageContent> const> contents, ::rust::Str condition, ::rust::Box<::loot::rust::Message> *return$) noexcept;

::loot::rust::MessageType loot$rust$cxxbridge1$Message$message_type(::loot::rust::Message const &self) noexcept;

::rust::repr::Fat loot$rust$cxxbridge1$Message$content(::loot::rust::Message const &self) noexcept;

::rust::repr::Fat loot$rust$cxxbridge1$Message$condition(::loot::rust::Message const &self) noexcept;

::loot::rust::Message *loot$rust$cxxbridge1$Message$boxed_clone(::loot::rust::Message const &self) noexcept;
::std::size_t loot$rust$cxxbridge1$MessageContent$operator$sizeof() noexcept;
::std::size_t loot$rust$cxxbridge1$MessageContent$operator$alignof() noexcept;

::rust::repr::Fat loot$rust$cxxbridge1$message_content_default_language() noexcept;

::loot::rust::MessageContent *loot$rust$cxxbridge1$new_message_content(::rust::String *text, ::rust::Str language) noexcept;

::rust::repr::Fat loot$rust$cxxbridge1$MessageContent$text(::loot::rust::MessageContent const &self) noexcept;

::rust::repr::Fat loot$rust$cxxbridge1$MessageContent$language(::loot::rust::MessageContent const &self) noexcept;

::loot::rust::MessageContent *loot$rust$cxxbridge1$MessageContent$boxed_clone(::loot::rust::MessageContent const &self) noexcept;
::std::size_t loot$rust$cxxbridge1$Group$operator$sizeof() noexcept;
::std::size_t loot$rust$cxxbridge1$Group$operator$alignof() noexcept;

::loot::rust::Group *loot$rust$cxxbridge1$new_group(::rust::String *name, ::rust::Str description, ::rust::Vec<::rust::String> *after_groups) noexcept;

::rust::repr::Fat loot$rust$cxxbridge1$group_default_name() noexcept;

::loot::rust::Group *loot$rust$cxxbridge1$Group$boxed_clone(::loot::rust::Group const &self) noexcept;

::rust::repr::Fat loot$rust$cxxbridge1$Group$name(::loot::rust::Group const &self) noexcept;

::rust::repr::Fat loot$rust$cxxbridge1$Group$description(::loot::rust::Group const &self) noexcept;

::rust::repr::Fat loot$rust$cxxbridge1$Group$after_groups(::loot::rust::Group const &self) noexcept;
::std::size_t loot$rust$cxxbridge1$Plugin$operator$sizeof() noexcept;
::std::size_t loot$rust$cxxbridge1$Plugin$operator$alignof() noexcept;

::rust::repr::Fat loot$rust$cxxbridge1$Plugin$name(::loot::rust::Plugin const &self) noexcept;

float loot$rust$cxxbridge1$Plugin$header_version(::loot::rust::Plugin const &self) noexcept;

::rust::repr::Fat loot$rust$cxxbridge1$Plugin$version(::loot::rust::Plugin const &self) noexcept;

::rust::repr::PtrLen loot$rust$cxxbridge1$Plugin$masters(::loot::rust::Plugin const &self, ::rust::Vec<::rust::String> *return$) noexcept;

::rust::repr::Fat loot$rust$cxxbridge1$Plugin$bash_tags(::loot::rust::Plugin const &self) noexcept;

::loot::rust::OptionalCrc *loot$rust$cxxbridge1$Plugin$crc(::loot::rust::Plugin const &self) noexcept;

bool loot$rust$cxxbridge1$Plugin$is_master(::loot::rust::Plugin const &self) noexcept;

bool loot$rust$cxxbridge1$Plugin$is_light_plugin(::loot::rust::Plugin const &self) noexcept;

bool loot$rust$cxxbridge1$Plugin$is_medium_plugin(::loot::rust::Plugin const &self) noexcept;

bool loot$rust$cxxbridge1$Plugin$is_update_plugin(::loot::rust::Plugin const &self) noexcept;

bool loot$rust$cxxbridge1$Plugin$is_blueprint_plugin(::loot::rust::Plugin const &self) noexcept;

::rust::repr::PtrLen loot$rust$cxxbridge1$Plugin$is_valid_as_light_plugin(::loot::rust::Plugin const &self, bool *return$) noexcept;

::rust::repr::PtrLen loot$rust$cxxbridge1$Plugin$is_valid_as_medium_plugin(::loot::rust::Plugin const &self, bool *return$) noexcept;

::rust::repr::PtrLen loot$rust$cxxbridge1$Plugin$is_valid_as_update_plugin(::loot::rust::Plugin const &self, bool *return$) noexcept;

bool loot$rust$cxxbridge1$Plugin$is_empty(::loot::rust::Plugin const &self) noexcept;

bool loot$rust$cxxbridge1$Plugin$loads_archive(::loot::rust::Plugin const &self) noexcept;

::rust::repr::PtrLen loot$rust$cxxbridge1$Plugin$do_records_overlap(::loot::rust::Plugin const &self, ::loot::rust::Plugin const &plugin, bool *return$) noexcept;

::loot::rust::Plugin *loot$rust$cxxbridge1$Plugin$boxed_clone(::loot::rust::Plugin const &self) noexcept;
::std::size_t loot$rust$cxxbridge1$OptionalPlugin$operator$sizeof() noexcept;
::std::size_t loot$rust$cxxbridge1$OptionalPlugin$operator$alignof() noexcept;

bool loot$rust$cxxbridge1$OptionalPlugin$is_some(::loot::rust::OptionalPlugin const &self) noexcept;

::rust::repr::PtrLen loot$rust$cxxbridge1$OptionalPlugin$as_ref(::loot::rust::OptionalPlugin const &self, ::loot::rust::Plugin const **return$) noexcept;
::std::size_t loot$rust$cxxbridge1$OptionalCrc$operator$sizeof() noexcept;
::std::size_t loot$rust$cxxbridge1$OptionalCrc$operator$alignof() noexcept;

bool loot$rust$cxxbridge1$OptionalCrc$is_some(::loot::rust::OptionalCrc const &self) noexcept;

::rust::repr::PtrLen loot$rust$cxxbridge1$OptionalCrc$as_ref(::loot::rust::OptionalCrc const &self, ::std::uint32_t const **return$) noexcept;
::std::size_t loot$rust$cxxbridge1$Vertex$operator$sizeof() noexcept;
::std::size_t loot$rust$cxxbridge1$Vertex$operator$alignof() noexcept;

::rust::repr::PtrLen loot$rust$cxxbridge1$new_vertex(::rust::String *name, ::loot::rust::EdgeType out_edge_type, ::rust::Box<::loot::rust::Vertex> *return$) noexcept;

::rust::repr::Fat loot$rust$cxxbridge1$Vertex$name(::loot::rust::Vertex const &self) noexcept;

::rust::repr::PtrLen loot$rust$cxxbridge1$Vertex$out_edge_type(::loot::rust::Vertex const &self, ::loot::rust::EdgeType *return$) noexcept;

::loot::rust::Vertex *loot$rust$cxxbridge1$Vertex$boxed_clone(::loot::rust::Vertex const &self) noexcept;
::std::size_t loot$rust$cxxbridge1$OptionalPluginMetadata$operator$sizeof() noexcept;
::std::size_t loot$rust$cxxbridge1$OptionalPluginMetadata$operator$alignof() noexcept;

bool loot$rust$cxxbridge1$OptionalPluginMetadata$is_some(::loot::rust::OptionalPluginMetadata const &self) noexcept;

::rust::repr::PtrLen loot$rust$cxxbridge1$OptionalPluginMetadata$as_ref(::loot::rust::OptionalPluginMetadata const &self, ::loot::rust::PluginMetadata const **return$) noexcept;
::std::size_t loot$rust$cxxbridge1$PluginMetadata$operator$sizeof() noexcept;
::std::size_t loot$rust$cxxbridge1$PluginMetadata$operator$alignof() noexcept;

::rust::repr::PtrLen loot$rust$cxxbridge1$new_plugin_metadata(::rust::Str name, ::rust::Box<::loot::rust::PluginMetadata> *return$) noexcept;

::rust::repr::Fat loot$rust$cxxbridge1$PluginMetadata$name(::loot::rust::PluginMetadata const &self) noexcept;

::rust::repr::Fat loot$rust$cxxbridge1$PluginMetadata$group(::loot::rust::PluginMetadata const &self) noexcept;

::rust::repr::Fat loot$rust$cxxbridge1$PluginMetadata$load_after_files(::loot::rust::PluginMetadata const &self) noexcept;

::rust::repr::Fat loot$rust$cxxbridge1$PluginMetadata$requirements(::loot::rust::PluginMetadata const &self) noexcept;

::rust::repr::Fat loot$rust$cxxbridge1$PluginMetadata$incompatibilities(::loot::rust::PluginMetadata const &self) noexcept;

::rust::repr::Fat loot$rust$cxxbridge1$PluginMetadata$messages(::loot::rust::PluginMetadata const &self) noexcept;

::rust::repr::Fat loot$rust$cxxbridge1$PluginMetadata$tags(::loot::rust::PluginMetadata const &self) noexcept;

::rust::repr::Fat loot$rust$cxxbridge1$PluginMetadata$dirty_info(::loot::rust::PluginMetadata const &self) noexcept;

::rust::repr::Fat loot$rust$cxxbridge1$PluginMetadata$clean_info(::loot::rust::PluginMetadata const &self) noexcept;

::rust::repr::Fat loot$rust$cxxbridge1$PluginMetadata$locations(::loot::rust::PluginMetadata const &self) noexcept;

void loot$rust$cxxbridge1$PluginMetadata$set_group(::loot::rust::PluginMetadata &self, ::rust::String *group) noexcept;

void loot$rust$cxxbridge1$PluginMetadata$unset_group(::loot::rust::PluginMetadata &self) noexcept;

void loot$rust$cxxbridge1$PluginMetadata$set_load_after_files(::loot::rust::PluginMetadata &self, ::rust::Slice<::rust::Box<::loot::rust::File> const> files) noexcept;

void loot$rust$cxxbridge1$PluginMetadata$set_requirements(::loot::rust::PluginMetadata &self, ::rust::Slice<::rust::Box<::loot::rust::File> const> files) noexcept;

void loot$rust$cxxbridge1$PluginMetadata$set_incompatibilities(::loot::rust::PluginMetadata &self, ::rust::Slice<::rust::Box<::loot::rust::File> const> files) noexcept;

void loot$rust$cxxbridge1$PluginMetadata$set_messages(::loot::rust::PluginMetadata &self, ::rust::Slice<::rust::Box<::loot::rust::Message> const> messages) noexcept;

void loot$rust$cxxbridge1$PluginMetadata$set_tags(::loot::rust::PluginMetadata &self, ::rust::Slice<::rust::Box<::loot::rust::Tag> const> tags) noexcept;

void loot$rust$cxxbridge1$PluginMetadata$set_dirty_info(::loot::rust::PluginMetadata &self, ::rust::Slice<::rust::Box<::loot::rust::PluginCleaningData> const> info) noexcept;

void loot$rust$cxxbridge1$PluginMetadata$set_clean_info(::loot::rust::PluginMetadata &self, ::rust::Slice<::rust::Box<::loot::rust::PluginCleaningData> const> info) noexcept;

void loot$rust$cxxbridge1$PluginMetadata$set_locations(::loot::rust::PluginMetadata &self, ::rust::Slice<::rust::Box<::loot::rust::Location> const> locations) noexcept;

void loot$rust$cxxbridge1$PluginMetadata$merge_metadata(::loot::rust::PluginMetadata &self, ::loot::rust::PluginMetadata const &plugin) noexcept;

bool loot$rust$cxxbridge1$PluginMetadata$has_name_only(::loot::rust::PluginMetadata const &self) noexcept;

bool loot$rust$cxxbridge1$PluginMetadata$is_regex_plugin(::loot::rust::PluginMetadata const &self) noexcept;

bool loot$rust$cxxbridge1$PluginMetadata$name_matches(::loot::rust::PluginMetadata const &self, ::rust::Str other_name) noexcept;

void loot$rust$cxxbridge1$PluginMetadata$as_yaml(::loot::rust::PluginMetadata const &self, ::rust::String *return$) noexcept;

::loot::rust::PluginMetadata *loot$rust$cxxbridge1$PluginMetadata$boxed_clone(::loot::rust::PluginMetadata const &self) noexcept;
::std::size_t loot$rust$cxxbridge1$File$operator$sizeof() noexcept;
::std::size_t loot$rust$cxxbridge1$File$operator$alignof() noexcept;

::rust::repr::PtrLen loot$rust$cxxbridge1$new_file(::rust::String *name, ::rust::Str display_name, ::rust::Str condition, ::rust::Slice<::rust::Box<::loot::rust::MessageContent> const> detail, ::rust::Str constraint, ::rust::Box<::loot::rust::File> *return$) noexcept;

::loot::rust::Filename const *loot$rust$cxxbridge1$File$filename(::loot::rust::File const &self) noexcept;

::rust::repr::Fat loot$rust$cxxbridge1$File$display_name(::loot::rust::File const &self) noexcept;

::rust::repr::Fat loot$rust$cxxbridge1$File$detail(::loot::rust::File const &self) noexcept;

::rust::repr::Fat loot$rust$cxxbridge1$File$condition(::loot::rust::File const &self) noexcept;

::rust::repr::Fat loot$rust$cxxbridge1$File$constraint(::loot::rust::File const &self) noexcept;

::loot::rust::File *loot$rust$cxxbridge1$File$boxed_clone(::loot::rust::File const &self) noexcept;
::std::size_t loot$rust$cxxbridge1$Filename$operator$sizeof() noexcept;
::std::size_t loot$rust$cxxbridge1$Filename$operator$alignof() noexcept;

::loot::rust::Filename *loot$rust$cxxbridge1$new_filename(::rust::String *name) noexcept;

::rust::repr::Fat loot$rust$cxxbridge1$Filename$as_str(::loot::rust::Filename const &self) noexcept;

::loot::rust::Filename *loot$rust$cxxbridge1$Filename$boxed_clone(::loot::rust::Filename const &self) noexcept;

bool loot$rust$cxxbridge1$Filename$eq(::loot::rust::Filename const &self, ::loot::rust::Filename const &other) noexcept;

bool loot$rust$cxxbridge1$Filename$ne(::loot::rust::Filename const &self, ::loot::rust::Filename const &other) noexcept;

bool loot$rust$cxxbridge1$Filename$lt(::loot::rust::Filename const &self, ::loot::rust::Filename const &other) noexcept;

bool loot$rust$cxxbridge1$Filename$le(::loot::rust::Filename const &self, ::loot::rust::Filename const &other) noexcept;

bool loot$rust$cxxbridge1$Filename$gt(::loot::rust::Filename const &self, ::loot::rust::Filename const &other) noexcept;

bool loot$rust$cxxbridge1$Filename$ge(::loot::rust::Filename const &self, ::loot::rust::Filename const &other) noexcept;
::std::size_t loot$rust$cxxbridge1$Tag$operator$sizeof() noexcept;
::std::size_t loot$rust$cxxbridge1$Tag$operator$alignof() noexcept;

::rust::repr::PtrLen loot$rust$cxxbridge1$new_tag(::rust::String *name, ::loot::rust::TagSuggestion suggestion, ::rust::Str condition, ::rust::Box<::loot::rust::Tag> *return$) noexcept;

::rust::repr::Fat loot$rust$cxxbridge1$Tag$name(::loot::rust::Tag const &self) noexcept;

bool loot$rust$cxxbridge1$Tag$is_addition(::loot::rust::Tag const &self) noexcept;

::rust::repr::Fat loot$rust$cxxbridge1$Tag$condition(::loot::rust::Tag const &self) noexcept;

::loot::rust::Tag *loot$rust$cxxbridge1$Tag$boxed_clone(::loot::rust::Tag const &self) noexcept;
::std::size_t loot$rust$cxxbridge1$PluginCleaningData$operator$sizeof() noexcept;
::std::size_t loot$rust$cxxbridge1$PluginCleaningData$operator$alignof() noexcept;

::rust::repr::PtrLen loot$rust$cxxbridge1$new_plugin_cleaning_data(::std::uint32_t crc, ::rust::String *cleaning_utility, ::rust::Slice<::rust::Box<::loot::rust::MessageContent> const> detail, ::std::uint32_t itm_count, ::std::uint32_t deleted_reference_count, ::std::uint32_t deleted_navmesh_count, ::rust::Box<::loot::rust::PluginCleaningData> *return$) noexcept;

::std::uint32_t loot$rust$cxxbridge1$PluginCleaningData$crc(::loot::rust::PluginCleaningData const &self) noexcept;

::std::uint32_t loot$rust$cxxbridge1$PluginCleaningData$itm_count(::loot::rust::PluginCleaningData const &self) noexcept;

::std::uint32_t loot$rust$cxxbridge1$PluginCleaningData$deleted_reference_count(::loot::rust::PluginCleaningData const &self) noexcept;

::std::uint32_t loot$rust$cxxbridge1$PluginCleaningData$deleted_navmesh_count(::loot::rust::PluginCleaningData const &self) noexcept;

::rust::repr::Fat loot$rust$cxxbridge1$PluginCleaningData$cleaning_utility(::loot::rust::PluginCleaningData const &self) noexcept;

::rust::repr::Fat loot$rust$cxxbridge1$PluginCleaningData$detail(::loot::rust::PluginCleaningData const &self) noexcept;

::loot::rust::PluginCleaningData *loot$rust$cxxbridge1$PluginCleaningData$boxed_clone(::loot::rust::PluginCleaningData const &self) noexcept;
::std::size_t loot$rust$cxxbridge1$Location$operator$sizeof() noexcept;
::std::size_t loot$rust$cxxbridge1$Location$operator$alignof() noexcept;

::loot::rust::Location *loot$rust$cxxbridge1$new_location(::rust::String *url, ::rust::Str name) noexcept;

::rust::repr::Fat loot$rust$cxxbridge1$Location$url(::loot::rust::Location const &self) noexcept;

::rust::repr::Fat loot$rust$cxxbridge1$Location$name(::loot::rust::Location const &self) noexcept;

::loot::rust::Location *loot$rust$cxxbridge1$Location$boxed_clone(::loot::rust::Location const &self) noexcept;
} // extern "C"

bool OptionalMessageContentRef::is_some() const noexcept {
  return loot$rust$cxxbridge1$OptionalMessageContentRef$is_some(*this);
}

::loot::rust::MessageContent const &OptionalMessageContentRef::as_ref() const {
  ::rust::MaybeUninit<::loot::rust::MessageContent const *> return$;
  ::rust::repr::PtrLen error$ = loot$rust$cxxbridge1$OptionalMessageContentRef$as_ref(*this, &return$.value);
  if (error$.ptr) {
    throw ::rust::impl<::rust::Error>::error(error$);
  }
  return *return$.value;
}

void set_log_level(::loot::rust::LogLevel level) {
  ::rust::repr::PtrLen error$ = loot$rust$cxxbridge1$set_log_level(level);
  if (error$.ptr) {
    throw ::rust::impl<::rust::Error>::error(error$);
  }
}

bool is_compatible(::std::uint32_t major, ::std::uint32_t minor, ::std::uint32_t patch) noexcept {
  return loot$rust$cxxbridge1$is_compatible(major, minor, patch);
}

::rust::String libloot_version() noexcept {
  ::rust::MaybeUninit<::rust::String> return$;
  loot$rust$cxxbridge1$libloot_version(&return$.value);
  return ::std::move(return$.value);
}

::rust::String libloot_revision() noexcept {
  ::rust::MaybeUninit<::rust::String> return$;
  loot$rust$cxxbridge1$libloot_revision(&return$.value);
  return ::std::move(return$.value);
}

::loot::rust::OptionalMessageContentRef select_message_content(::rust::Slice<::loot::rust::MessageContent const> contents, ::rust::Str language) noexcept {
  ::rust::MaybeUninit<::loot::rust::OptionalMessageContentRef> return$;
  loot$rust$cxxbridge1$select_message_content(contents, language, &return$.value);
  return ::std::move(return$.value);
}

::std::size_t Game::layout::size() noexcept {
  return loot$rust$cxxbridge1$Game$operator$sizeof();
}

::std::size_t Game::layout::align() noexcept {
  return loot$rust$cxxbridge1$Game$operator$alignof();
}

::rust::Box<::loot::rust::Game> new_game(::loot::rust::GameType game_type, ::rust::Str game_path) {
  ::rust::MaybeUninit<::rust::Box<::loot::rust::Game>> return$;
  ::rust::repr::PtrLen error$ = loot$rust$cxxbridge1$new_game(game_type, game_path, &return$.value);
  if (error$.ptr) {
    throw ::rust::impl<::rust::Error>::error(error$);
  }
  return ::std::move(return$.value);
}

::rust::Box<::loot::rust::Game> new_game_with_local_path(::loot::rust::GameType game_type, ::rust::Str game_path, ::rust::Str game_local_path) {
  ::rust::MaybeUninit<::rust::Box<::loot::rust::Game>> return$;
  ::rust::repr::PtrLen error$ = loot$rust$cxxbridge1$new_game_with_local_path(game_type, game_path, game_local_path, &return$.value);
  if (error$.ptr) {
    throw ::rust::impl<::rust::Error>::error(error$);
  }
  return ::std::move(return$.value);
}

::loot::rust::GameType Game::game_type() const {
  ::rust::MaybeUninit<::loot::rust::GameType> return$;
  ::rust::repr::PtrLen error$ = loot$rust$cxxbridge1$Game$game_type(*this, &return$.value);
  if (error$.ptr) {
    throw ::rust::impl<::rust::Error>::error(error$);
  }
  return ::std::move(return$.value);
}

::rust::Vec<::rust::String> Game::additional_data_paths() const {
  ::rust::MaybeUninit<::rust::Vec<::rust::String>> return$;
  ::rust::repr::PtrLen error$ = loot$rust$cxxbridge1$Game$additional_data_paths(*this, &return$.value);
  if (error$.ptr) {
    throw ::rust::impl<::rust::Error>::error(error$);
  }
  return ::std::move(return$.value);
}

void Game::set_additional_data_paths(::rust::Slice<::rust::Str const> additional_data_paths) {
  ::rust::repr::PtrLen error$ = loot$rust$cxxbridge1$Game$set_additional_data_paths(*this, additional_data_paths);
  if (error$.ptr) {
    throw ::rust::impl<::rust::Error>::error(error$);
  }
}

::rust::Box<::loot::rust::Database> Game::database() const noexcept {
  return ::rust::Box<::loot::rust::Database>::from_raw(loot$rust$cxxbridge1$Game$database(*this));
}

bool Game::is_valid_plugin(::rust::Str plugin_path) const noexcept {
  return loot$rust$cxxbridge1$Game$is_valid_plugin(*this, plugin_path);
}

void Game::load_plugins(::rust::Slice<::rust::Str const> plugin_paths) {
  ::rust::repr::PtrLen error$ = loot$rust$cxxbridge1$Game$load_plugins(*this, plugin_paths);
  if (error$.ptr) {
    throw ::rust::impl<::rust::Error>::error(error$);
  }
}

void Game::load_plugin_headers(::rust::Slice<::rust::Str const> plugin_paths) {
  ::rust::repr::PtrLen error$ = loot$rust$cxxbridge1$Game$load_plugin_headers(*this, plugin_paths);
  if (error$.ptr) {
    throw ::rust::impl<::rust::Error>::error(error$);
  }
}

void Game::clear_loaded_plugins() noexcept {
  loot$rust$cxxbridge1$Game$clear_loaded_plugins(*this);
}

::rust::Box<::loot::rust::OptionalPlugin> Game::plugin(::rust::Str plugin_name) const noexcept {
  return ::rust::Box<::loot::rust::OptionalPlugin>::from_raw(loot$rust$cxxbridge1$Game$plugin(*this, plugin_name));
}

::rust::Vec<::loot::rust::Plugin> Game::loaded_plugins() const noexcept {
  ::rust::MaybeUninit<::rust::Vec<::loot::rust::Plugin>> return$;
  loot$rust$cxxbridge1$Game$loaded_plugins(*this, &return$.value);
  return ::std::move(return$.value);
}

::rust::Vec<::rust::String> Game::sort_plugins(::rust::Slice<::rust::Str const> plugin_names) const {
  ::rust::MaybeUninit<::rust::Vec<::rust::String>> return$;
  ::rust::repr::PtrLen error$ = loot$rust$cxxbridge1$Game$sort_plugins(*this, plugin_names, &return$.value);
  if (error$.ptr) {
    throw ::rust::impl<::rust::Error>::error(error$);
  }
  return ::std::move(return$.value);
}

void Game::load_current_load_order_state() {
  ::rust::repr::PtrLen error$ = loot$rust$cxxbridge1$Game$load_current_load_order_state(*this);
  if (error$.ptr) {
    throw ::rust::impl<::rust::Error>::error(error$);
  }
}

bool Game::is_load_order_ambiguous() const {
  ::rust::MaybeUninit<bool> return$;
  ::rust::repr::PtrLen error$ = loot$rust$cxxbridge1$Game$is_load_order_ambiguous(*this, &return$.value);
  if (error$.ptr) {
    throw ::rust::impl<::rust::Error>::error(error$);
  }
  return ::std::move(return$.value);
}

::rust::String Game::active_plugins_file_path() const {
  ::rust::MaybeUninit<::rust::String> return$;
  ::rust::repr::PtrLen error$ = loot$rust$cxxbridge1$Game$active_plugins_file_path(*this, &return$.value);
  if (error$.ptr) {
    throw ::rust::impl<::rust::Error>::error(error$);
  }
  return ::std::move(return$.value);
}

bool Game::is_plugin_active(::rust::Str plugin_name) const noexcept {
  return loot$rust$cxxbridge1$Game$is_plugin_active(*this, plugin_name);
}

::rust::Vec<::rust::String> Game::load_order() const noexcept {
  ::rust::MaybeUninit<::rust::Vec<::rust::String>> return$;
  loot$rust$cxxbridge1$Game$load_order(*this, &return$.value);
  return ::std::move(return$.value);
}

void Game::set_load_order(::rust::Slice<::rust::Str const> load_order) {
  ::rust::repr::PtrLen error$ = loot$rust$cxxbridge1$Game$set_load_order(*this, load_order);
  if (error$.ptr) {
    throw ::rust::impl<::rust::Error>::error(error$);
  }
}

::std::size_t Database::layout::size() noexcept {
  return loot$rust$cxxbridge1$Database$operator$sizeof();
}

::std::size_t Database::layout::align() noexcept {
  return loot$rust$cxxbridge1$Database$operator$alignof();
}

void Database::load_masterlist(::rust::Str path) const {
  ::rust::repr::PtrLen error$ = loot$rust$cxxbridge1$Database$load_masterlist(*this, path);
  if (error$.ptr) {
    throw ::rust::impl<::rust::Error>::error(error$);
  }
}

void Database::load_masterlist_with_prelude(::rust::Str masterlist_path, ::rust::Str prelude_path) const {
  ::rust::repr::PtrLen error$ = loot$rust$cxxbridge1$Database$load_masterlist_with_prelude(*this, masterlist_path, prelude_path);
  if (error$.ptr) {
    throw ::rust::impl<::rust::Error>::error(error$);
  }
}

void Database::load_userlist(::rust::Str path) const {
  ::rust::repr::PtrLen error$ = loot$rust$cxxbridge1$Database$load_userlist(*this, path);
  if (error$.ptr) {
    throw ::rust::impl<::rust::Error>::error(error$);
  }
}

void Database::write_user_metadata(::rust::Str output_path, bool overwrite) const {
  ::rust::repr::PtrLen error$ = loot$rust$cxxbridge1$Database$write_user_metadata(*this, output_path, overwrite);
  if (error$.ptr) {
    throw ::rust::impl<::rust::Error>::error(error$);
  }
}

void Database::write_minimal_list(::rust::Str output_path, bool overwrite) const {
  ::rust::repr::PtrLen error$ = loot$rust$cxxbridge1$Database$write_minimal_list(*this, output_path, overwrite);
  if (error$.ptr) {
    throw ::rust::impl<::rust::Error>::error(error$);
  }
}

bool Database::evaluate(::rust::Str condition) const {
  ::rust::MaybeUninit<bool> return$;
  ::rust::repr::PtrLen error$ = loot$rust$cxxbridge1$Database$evaluate(*this, condition, &return$.value);
  if (error$.ptr) {
    throw ::rust::impl<::rust::Error>::error(error$);
  }
  return ::std::move(return$.value);
}

::rust::Vec<::rust::String> Database::known_bash_tags() const {
  ::rust::MaybeUninit<::rust::Vec<::rust::String>> return$;
  ::rust::repr::PtrLen error$ = loot$rust$cxxbridge1$Database$known_bash_tags(*this, &return$.value);
  if (error$.ptr) {
    throw ::rust::impl<::rust::Error>::error(error$);
  }
  return ::std::move(return$.value);
}

::rust::Vec<::loot::rust::Message> Database::general_messages(bool evaluate_conditions) const {
  ::rust::MaybeUninit<::rust::Vec<::loot::rust::Message>> return$;
  ::rust::repr::PtrLen error$ = loot$rust$cxxbridge1$Database$general_messages(*this, evaluate_conditions, &return$.value);
  if (error$.ptr) {
    throw ::rust::impl<::rust::Error>::error(error$);
  }
  return ::std::move(return$.value);
}

::rust::Vec<::loot::rust::Group> Database::groups(bool include_user_metadata) const {
  ::rust::MaybeUninit<::rust::Vec<::loot::rust::Group>> return$;
  ::rust::repr::PtrLen error$ = loot$rust$cxxbridge1$Database$groups(*this, include_user_metadata, &return$.value);
  if (error$.ptr) {
    throw ::rust::impl<::rust::Error>::error(error$);
  }
  return ::std::move(return$.value);
}

::rust::Vec<::loot::rust::Group> Database::user_groups() const {
  ::rust::MaybeUninit<::rust::Vec<::loot::rust::Group>> return$;
  ::rust::repr::PtrLen error$ = loot$rust$cxxbridge1$Database$user_groups(*this, &return$.value);
  if (error$.ptr) {
    throw ::rust::impl<::rust::Error>::error(error$);
  }
  return ::std::move(return$.value);
}

void Database::set_user_groups(::rust::Slice<::rust::Box<::loot::rust::Group> const> groups) const {
  ::rust::repr::PtrLen error$ = loot$rust$cxxbridge1$Database$set_user_groups(*this, groups);
  if (error$.ptr) {
    throw ::rust::impl<::rust::Error>::error(error$);
  }
}

::rust::Vec<::loot::rust::Vertex> Database::groups_path(::rust::Str from_group_name, ::rust::Str to_group_name) const {
  ::rust::MaybeUninit<::rust::Vec<::loot::rust::Vertex>> return$;
  ::rust::repr::PtrLen error$ = loot$rust$cxxbridge1$Database$groups_path(*this, from_group_name, to_group_name, &return$.value);
  if (error$.ptr) {
    throw ::rust::impl<::rust::Error>::error(error$);
  }
  return ::std::move(return$.value);
}

::rust::Box<::loot::rust::OptionalPluginMetadata> Database::plugin_metadata(::rust::Str plugin_name, bool include_user_metadata, bool evaluate_conditions) const {
  ::rust::MaybeUninit<::rust::Box<::loot::rust::OptionalPluginMetadata>> return$;
  ::rust::repr::PtrLen error$ = loot$rust$cxxbridge1$Database$plugin_metadata(*this, plugin_name, include_user_metadata, evaluate_conditions, &return$.value);
  if (error$.ptr) {
    throw ::rust::impl<::rust::Error>::error(error$);
  }
  return ::std::move(return$.value);
}

::rust::Box<::loot::rust::OptionalPluginMetadata> Database::plugin_user_metadata(::rust::Str plugin_name, bool evaluate_conditions) const {
  ::rust::MaybeUninit<::rust::Box<::loot::rust::OptionalPluginMetadata>> return$;
  ::rust::repr::PtrLen error$ = loot$rust$cxxbridge1$Database$plugin_user_metadata(*this, plugin_name, evaluate_conditions, &return$.value);
  if (error$.ptr) {
    throw ::rust::impl<::rust::Error>::error(error$);
  }
  return ::std::move(return$.value);
}

void Database::set_plugin_user_metadata(::rust::Box<::loot::rust::PluginMetadata> plugin_metadata) {
  ::rust::repr::PtrLen error$ = loot$rust$cxxbridge1$Database$set_plugin_user_metadata(*this, plugin_metadata.into_raw());
  if (error$.ptr) {
    throw ::rust::impl<::rust::Error>::error(error$);
  }
}

void Database::discard_plugin_user_metadata(::rust::Str plugin) const {
  ::rust::repr::PtrLen error$ = loot$rust$cxxbridge1$Database$discard_plugin_user_metadata(*this, plugin);
  if (error$.ptr) {
    throw ::rust::impl<::rust::Error>::error(error$);
  }
}

void Database::discard_all_user_metadata() const {
  ::rust::repr::PtrLen error$ = loot$rust$cxxbridge1$Database$discard_all_user_metadata(*this);
  if (error$.ptr) {
    throw ::rust::impl<::rust::Error>::error(error$);
  }
}

::std::size_t Message::layout::size() noexcept {
  return loot$rust$cxxbridge1$Message$operator$sizeof();
}

::std::size_t Message::layout::align() noexcept {
  return loot$rust$cxxbridge1$Message$operator$alignof();
}

::rust::Box<::loot::rust::Message> new_message(::loot::rust::MessageType message_type, ::rust::String content, ::rust::Str condition) {
  ::rust::MaybeUninit<::rust::Box<::loot::rust::Message>> return$;
  ::rust::repr::PtrLen error$ = loot$rust$cxxbridge1$new_message(message_type, &content, condition, &return$.value);
  if (error$.ptr) {
    throw ::rust::impl<::rust::Error>::error(error$);
  }
  return ::std::move(return$.value);
}

::rust::Box<::loot::rust::Message> multilingual_message(::loot::rust::MessageType message_type, ::rust::Slice<::rust::Box<::loot::rust::MessageContent> const> contents, ::rust::Str condition) {
  ::rust::MaybeUninit<::rust::Box<::loot::rust::Message>> return$;
  ::rust::repr::PtrLen error$ = loot$rust$cxxbridge1$multilingual_message(message_type, contents, condition, &return$.value);
  if (error$.ptr) {
    throw ::rust::impl<::rust::Error>::error(error$);
  }
  return ::std::move(return$.value);
}

::loot::rust::MessageType Message::message_type() const noexcept {
  return loot$rust$cxxbridge1$Message$message_type(*this);
}

::rust::Slice<::loot::rust::MessageContent const> Message::content() const noexcept {
  return ::rust::impl<::rust::Slice<::loot::rust::MessageContent const>>::slice(loot$rust$cxxbridge1$Message$content(*this));
}

::rust::Str Message::condition() const noexcept {
  return ::rust::impl<::rust::Str>::new_unchecked(loot$rust$cxxbridge1$Message$condition(*this));
}

::rust::Box<::loot::rust::Message> Message::boxed_clone() const noexcept {
  return ::rust::Box<::loot::rust::Message>::from_raw(loot$rust$cxxbridge1$Message$boxed_clone(*this));
}

::std::size_t MessageContent::layout::size() noexcept {
  return loot$rust$cxxbridge1$MessageContent$operator$sizeof();
}

::std::size_t MessageContent::layout::align() noexcept {
  return loot$rust$cxxbridge1$MessageContent$operator$alignof();
}

::rust::Str message_content_default_language() noexcept {
  return ::rust::impl<::rust::Str>::new_unchecked(loot$rust$cxxbridge1$message_content_default_language());
}

::rust::Box<::loot::rust::MessageContent> new_message_content(::rust::String text, ::rust::Str language) noexcept {
  return ::rust::Box<::loot::rust::MessageContent>::from_raw(loot$rust$cxxbridge1$new_message_content(&text, language));
}

::rust::Str MessageContent::text() const noexcept {
  return ::rust::impl<::rust::Str>::new_unchecked(loot$rust$cxxbridge1$MessageContent$text(*this));
}

::rust::Str MessageContent::language() const noexcept {
  return ::rust::impl<::rust::Str>::new_unchecked(loot$rust$cxxbridge1$MessageContent$language(*this));
}

::rust::Box<::loot::rust::MessageContent> MessageContent::boxed_clone() const noexcept {
  return ::rust::Box<::loot::rust::MessageContent>::from_raw(loot$rust$cxxbridge1$MessageContent$boxed_clone(*this));
}

::std::size_t Group::layout::size() noexcept {
  return loot$rust$cxxbridge1$Group$operator$sizeof();
}

::std::size_t Group::layout::align() noexcept {
  return loot$rust$cxxbridge1$Group$operator$alignof();
}

::rust::Box<::loot::rust::Group> new_group(::rust::String name, ::rust::Str description, ::rust::Vec<::rust::String> after_groups) noexcept {
  ::rust::ManuallyDrop<::rust::Vec<::rust::String>> after_groups$(::std::move(after_groups));
  return ::rust::Box<::loot::rust::Group>::from_raw(loot$rust$cxxbridge1$new_group(&name, description, &after_groups$.value));
}

::rust::Str group_default_name() noexcept {
  return ::rust::impl<::rust::Str>::new_unchecked(loot$rust$cxxbridge1$group_default_name());
}

::rust::Box<::loot::rust::Group> Group::boxed_clone() const noexcept {
  return ::rust::Box<::loot::rust::Group>::from_raw(loot$rust$cxxbridge1$Group$boxed_clone(*this));
}

::rust::Str Group::name() const noexcept {
  return ::rust::impl<::rust::Str>::new_unchecked(loot$rust$cxxbridge1$Group$name(*this));
}

::rust::Str Group::description() const noexcept {
  return ::rust::impl<::rust::Str>::new_unchecked(loot$rust$cxxbridge1$Group$description(*this));
}

::rust::Slice<::rust::String const> Group::after_groups() const noexcept {
  return ::rust::impl<::rust::Slice<::rust::String const>>::slice(loot$rust$cxxbridge1$Group$after_groups(*this));
}

::std::size_t Plugin::layout::size() noexcept {
  return loot$rust$cxxbridge1$Plugin$operator$sizeof();
}

::std::size_t Plugin::layout::align() noexcept {
  return loot$rust$cxxbridge1$Plugin$operator$alignof();
}

::rust::Str Plugin::name() const noexcept {
  return ::rust::impl<::rust::Str>::new_unchecked(loot$rust$cxxbridge1$Plugin$name(*this));
}

float Plugin::header_version() const noexcept {
  return loot$rust$cxxbridge1$Plugin$header_version(*this);
}

::rust::Str Plugin::version() const noexcept {
  return ::rust::impl<::rust::Str>::new_unchecked(loot$rust$cxxbridge1$Plugin$version(*this));
}

::rust::Vec<::rust::String> Plugin::masters() const {
  ::rust::MaybeUninit<::rust::Vec<::rust::String>> return$;
  ::rust::repr::PtrLen error$ = loot$rust$cxxbridge1$Plugin$masters(*this, &return$.value);
  if (error$.ptr) {
    throw ::rust::impl<::rust::Error>::error(error$);
  }
  return ::std::move(return$.value);
}

::rust::Slice<::rust::String const> Plugin::bash_tags() const noexcept {
  return ::rust::impl<::rust::Slice<::rust::String const>>::slice(loot$rust$cxxbridge1$Plugin$bash_tags(*this));
}

::rust::Box<::loot::rust::OptionalCrc> Plugin::crc() const noexcept {
  return ::rust::Box<::loot::rust::OptionalCrc>::from_raw(loot$rust$cxxbridge1$Plugin$crc(*this));
}

bool Plugin::is_master() const noexcept {
  return loot$rust$cxxbridge1$Plugin$is_master(*this);
}

bool Plugin::is_light_plugin() const noexcept {
  return loot$rust$cxxbridge1$Plugin$is_light_plugin(*this);
}

bool Plugin::is_medium_plugin() const noexcept {
  return loot$rust$cxxbridge1$Plugin$is_medium_plugin(*this);
}

bool Plugin::is_update_plugin() const noexcept {
  return loot$rust$cxxbridge1$Plugin$is_update_plugin(*this);
}

bool Plugin::is_blueprint_plugin() const noexcept {
  return loot$rust$cxxbridge1$Plugin$is_blueprint_plugin(*this);
}

bool Plugin::is_valid_as_light_plugin() const {
  ::rust::MaybeUninit<bool> return$;
  ::rust::repr::PtrLen error$ = loot$rust$cxxbridge1$Plugin$is_valid_as_light_plugin(*this, &return$.value);
  if (error$.ptr) {
    throw ::rust::impl<::rust::Error>::error(error$);
  }
  return ::std::move(return$.value);
}

bool Plugin::is_valid_as_medium_plugin() const {
  ::rust::MaybeUninit<bool> return$;
  ::rust::repr::PtrLen error$ = loot$rust$cxxbridge1$Plugin$is_valid_as_medium_plugin(*this, &return$.value);
  if (error$.ptr) {
    throw ::rust::impl<::rust::Error>::error(error$);
  }
  return ::std::move(return$.value);
}

bool Plugin::is_valid_as_update_plugin() const {
  ::rust::MaybeUninit<bool> return$;
  ::rust::repr::PtrLen error$ = loot$rust$cxxbridge1$Plugin$is_valid_as_update_plugin(*this, &return$.value);
  if (error$.ptr) {
    throw ::rust::impl<::rust::Error>::error(error$);
  }
  return ::std::move(return$.value);
}

bool Plugin::is_empty() const noexcept {
  return loot$rust$cxxbridge1$Plugin$is_empty(*this);
}

bool Plugin::loads_archive() const noexcept {
  return loot$rust$cxxbridge1$Plugin$loads_archive(*this);
}

bool Plugin::do_records_overlap(::loot::rust::Plugin const &plugin) const {
  ::rust::MaybeUninit<bool> return$;
  ::rust::repr::PtrLen error$ = loot$rust$cxxbridge1$Plugin$do_records_overlap(*this, plugin, &return$.value);
  if (error$.ptr) {
    throw ::rust::impl<::rust::Error>::error(error$);
  }
  return ::std::move(return$.value);
}

::rust::Box<::loot::rust::Plugin> Plugin::boxed_clone() const noexcept {
  return ::rust::Box<::loot::rust::Plugin>::from_raw(loot$rust$cxxbridge1$Plugin$boxed_clone(*this));
}

::std::size_t OptionalPlugin::layout::size() noexcept {
  return loot$rust$cxxbridge1$OptionalPlugin$operator$sizeof();
}

::std::size_t OptionalPlugin::layout::align() noexcept {
  return loot$rust$cxxbridge1$OptionalPlugin$operator$alignof();
}

bool OptionalPlugin::is_some() const noexcept {
  return loot$rust$cxxbridge1$OptionalPlugin$is_some(*this);
}

::loot::rust::Plugin const &OptionalPlugin::as_ref() const {
  ::rust::MaybeUninit<::loot::rust::Plugin const *> return$;
  ::rust::repr::PtrLen error$ = loot$rust$cxxbridge1$OptionalPlugin$as_ref(*this, &return$.value);
  if (error$.ptr) {
    throw ::rust::impl<::rust::Error>::error(error$);
  }
  return *return$.value;
}

::std::size_t OptionalCrc::layout::size() noexcept {
  return loot$rust$cxxbridge1$OptionalCrc$operator$sizeof();
}

::std::size_t OptionalCrc::layout::align() noexcept {
  return loot$rust$cxxbridge1$OptionalCrc$operator$alignof();
}

bool OptionalCrc::is_some() const noexcept {
  return loot$rust$cxxbridge1$OptionalCrc$is_some(*this);
}

::std::uint32_t const &OptionalCrc::as_ref() const {
  ::rust::MaybeUninit<::std::uint32_t const *> return$;
  ::rust::repr::PtrLen error$ = loot$rust$cxxbridge1$OptionalCrc$as_ref(*this, &return$.value);
  if (error$.ptr) {
    throw ::rust::impl<::rust::Error>::error(error$);
  }
  return *return$.value;
}

::std::size_t Vertex::layout::size() noexcept {
  return loot$rust$cxxbridge1$Vertex$operator$sizeof();
}

::std::size_t Vertex::layout::align() noexcept {
  return loot$rust$cxxbridge1$Vertex$operator$alignof();
}

::rust::Box<::loot::rust::Vertex> new_vertex(::rust::String name, ::loot::rust::EdgeType out_edge_type) {
  ::rust::MaybeUninit<::rust::Box<::loot::rust::Vertex>> return$;
  ::rust::repr::PtrLen error$ = loot$rust$cxxbridge1$new_vertex(&name, out_edge_type, &return$.value);
  if (error$.ptr) {
    throw ::rust::impl<::rust::Error>::error(error$);
  }
  return ::std::move(return$.value);
}

::rust::Str Vertex::name() const noexcept {
  return ::rust::impl<::rust::Str>::new_unchecked(loot$rust$cxxbridge1$Vertex$name(*this));
}

::loot::rust::EdgeType Vertex::out_edge_type() const {
  ::rust::MaybeUninit<::loot::rust::EdgeType> return$;
  ::rust::repr::PtrLen error$ = loot$rust$cxxbridge1$Vertex$out_edge_type(*this, &return$.value);
  if (error$.ptr) {
    throw ::rust::impl<::rust::Error>::error(error$);
  }
  return ::std::move(return$.value);
}

::rust::Box<::loot::rust::Vertex> Vertex::boxed_clone() const noexcept {
  return ::rust::Box<::loot::rust::Vertex>::from_raw(loot$rust$cxxbridge1$Vertex$boxed_clone(*this));
}

::std::size_t OptionalPluginMetadata::layout::size() noexcept {
  return loot$rust$cxxbridge1$OptionalPluginMetadata$operator$sizeof();
}

::std::size_t OptionalPluginMetadata::layout::align() noexcept {
  return loot$rust$cxxbridge1$OptionalPluginMetadata$operator$alignof();
}

bool OptionalPluginMetadata::is_some() const noexcept {
  return loot$rust$cxxbridge1$OptionalPluginMetadata$is_some(*this);
}

::loot::rust::PluginMetadata const &OptionalPluginMetadata::as_ref() const {
  ::rust::MaybeUninit<::loot::rust::PluginMetadata const *> return$;
  ::rust::repr::PtrLen error$ = loot$rust$cxxbridge1$OptionalPluginMetadata$as_ref(*this, &return$.value);
  if (error$.ptr) {
    throw ::rust::impl<::rust::Error>::error(error$);
  }
  return *return$.value;
}

::std::size_t PluginMetadata::layout::size() noexcept {
  return loot$rust$cxxbridge1$PluginMetadata$operator$sizeof();
}

::std::size_t PluginMetadata::layout::align() noexcept {
  return loot$rust$cxxbridge1$PluginMetadata$operator$alignof();
}

::rust::Box<::loot::rust::PluginMetadata> new_plugin_metadata(::rust::Str name) {
  ::rust::MaybeUninit<::rust::Box<::loot::rust::PluginMetadata>> return$;
  ::rust::repr::PtrLen error$ = loot$rust$cxxbridge1$new_plugin_metadata(name, &return$.value);
  if (error$.ptr) {
    throw ::rust::impl<::rust::Error>::error(error$);
  }
  return ::std::move(return$.value);
}

::rust::Str PluginMetadata::name() const noexcept {
  return ::rust::impl<::rust::Str>::new_unchecked(loot$rust$cxxbridge1$PluginMetadata$name(*this));
}

::rust::Str PluginMetadata::group() const noexcept {
  return ::rust::impl<::rust::Str>::new_unchecked(loot$rust$cxxbridge1$PluginMetadata$group(*this));
}

::rust::Slice<::loot::rust::File const> PluginMetadata::load_after_files() const noexcept {
  return ::rust::impl<::rust::Slice<::loot::rust::File const>>::slice(loot$rust$cxxbridge1$PluginMetadata$load_after_files(*this));
}

::rust::Slice<::loot::rust::File const> PluginMetadata::requirements() const noexcept {
  return ::rust::impl<::rust::Slice<::loot::rust::File const>>::slice(loot$rust$cxxbridge1$PluginMetadata$requirements(*this));
}

::rust::Slice<::loot::rust::File const> PluginMetadata::incompatibilities() const noexcept {
  return ::rust::impl<::rust::Slice<::loot::rust::File const>>::slice(loot$rust$cxxbridge1$PluginMetadata$incompatibilities(*this));
}

::rust::Slice<::loot::rust::Message const> PluginMetadata::messages() const noexcept {
  return ::rust::impl<::rust::Slice<::loot::rust::Message const>>::slice(loot$rust$cxxbridge1$PluginMetadata$messages(*this));
}

::rust::Slice<::loot::rust::Tag const> PluginMetadata::tags() const noexcept {
  return ::rust::impl<::rust::Slice<::loot::rust::Tag const>>::slice(loot$rust$cxxbridge1$PluginMetadata$tags(*this));
}

::rust::Slice<::loot::rust::PluginCleaningData const> PluginMetadata::dirty_info() const noexcept {
  return ::rust::impl<::rust::Slice<::loot::rust::PluginCleaningData const>>::slice(loot$rust$cxxbridge1$PluginMetadata$dirty_info(*this));
}

::rust::Slice<::loot::rust::PluginCleaningData const> PluginMetadata::clean_info() const noexcept {
  return ::rust::impl<::rust::Slice<::loot::rust::PluginCleaningData const>>::slice(loot$rust$cxxbridge1$PluginMetadata$clean_info(*this));
}

::rust::Slice<::loot::rust::Location const> PluginMetadata::locations() const noexcept {
  return ::rust::impl<::rust::Slice<::loot::rust::Location const>>::slice(loot$rust$cxxbridge1$PluginMetadata$locations(*this));
}

void PluginMetadata::set_group(::rust::String group) noexcept {
  loot$rust$cxxbridge1$PluginMetadata$set_group(*this, &group);
}

void PluginMetadata::unset_group() noexcept {
  loot$rust$cxxbridge1$PluginMetadata$unset_group(*this);
}

void PluginMetadata::set_load_after_files(::rust::Slice<::rust::Box<::loot::rust::File> const> files) noexcept {
  loot$rust$cxxbridge1$PluginMetadata$set_load_after_files(*this, files);
}

void PluginMetadata::set_requirements(::rust::Slice<::rust::Box<::loot::rust::File> const> files) noexcept {
  loot$rust$cxxbridge1$PluginMetadata$set_requirements(*this, files);
}

void PluginMetadata::set_incompatibilities(::rust::Slice<::rust::Box<::loot::rust::File> const> files) noexcept {
  loot$rust$cxxbridge1$PluginMetadata$set_incompatibilities(*this, files);
}

void PluginMetadata::set_messages(::rust::Slice<::rust::Box<::loot::rust::Message> const> messages) noexcept {
  loot$rust$cxxbridge1$PluginMetadata$set_messages(*this, messages);
}

void PluginMetadata::set_tags(::rust::Slice<::rust::Box<::loot::rust::Tag> const> tags) noexcept {
  loot$rust$cxxbridge1$PluginMetadata$set_tags(*this, tags);
}

void PluginMetadata::set_dirty_info(::rust::Slice<::rust::Box<::loot::rust::PluginCleaningData> const> info) noexcept {
  loot$rust$cxxbridge1$PluginMetadata$set_dirty_info(*this, info);
}

void PluginMetadata::set_clean_info(::rust::Slice<::rust::Box<::loot::rust::PluginCleaningData> const> info) noexcept {
  loot$rust$cxxbridge1$PluginMetadata$set_clean_info(*this, info);
}

void PluginMetadata::set_locations(::rust::Slice<::rust::Box<::loot::rust::Location> const> locations) noexcept {
  loot$rust$cxxbridge1$PluginMetadata$set_locations(*this, locations);
}

void PluginMetadata::merge_metadata(::loot::rust::PluginMetadata const &plugin) noexcept {
  loot$rust$cxxbridge1$PluginMetadata$merge_metadata(*this, plugin);
}

bool PluginMetadata::has_name_only() const noexcept {
  return loot$rust$cxxbridge1$PluginMetadata$has_name_only(*this);
}

bool PluginMetadata::is_regex_plugin() const noexcept {
  return loot$rust$cxxbridge1$PluginMetadata$is_regex_plugin(*this);
}

bool PluginMetadata::name_matches(::rust::Str other_name) const noexcept {
  return loot$rust$cxxbridge1$PluginMetadata$name_matches(*this, other_name);
}

::rust::String PluginMetadata::as_yaml() const noexcept {
  ::rust::MaybeUninit<::rust::String> return$;
  loot$rust$cxxbridge1$PluginMetadata$as_yaml(*this, &return$.value);
  return ::std::move(return$.value);
}

::rust::Box<::loot::rust::PluginMetadata> PluginMetadata::boxed_clone() const noexcept {
  return ::rust::Box<::loot::rust::PluginMetadata>::from_raw(loot$rust$cxxbridge1$PluginMetadata$boxed_clone(*this));
}

::std::size_t File::layout::size() noexcept {
  return loot$rust$cxxbridge1$File$operator$sizeof();
}

::std::size_t File::layout::align() noexcept {
  return loot$rust$cxxbridge1$File$operator$alignof();
}

::rust::Box<::loot::rust::File> new_file(::rust::String name, ::rust::Str display_name, ::rust::Str condition, ::rust::Slice<::rust::Box<::loot::rust::MessageContent> const> detail, ::rust::Str constraint) {
  ::rust::MaybeUninit<::rust::Box<::loot::rust::File>> return$;
  ::rust::repr::PtrLen error$ = loot$rust$cxxbridge1$new_file(&name, display_name, condition, detail, constraint, &return$.value);
  if (error$.ptr) {
    throw ::rust::impl<::rust::Error>::error(error$);
  }
  return ::std::move(return$.value);
}

::loot::rust::Filename const &File::filename() const noexcept {
  return *loot$rust$cxxbridge1$File$filename(*this);
}

::rust::Str File::display_name() const noexcept {
  return ::rust::impl<::rust::Str>::new_unchecked(loot$rust$cxxbridge1$File$display_name(*this));
}

::rust::Slice<::loot::rust::MessageContent const> File::detail() const noexcept {
  return ::rust::impl<::rust::Slice<::loot::rust::MessageContent const>>::slice(loot$rust$cxxbridge1$File$detail(*this));
}

::rust::Str File::condition() const noexcept {
  return ::rust::impl<::rust::Str>::new_unchecked(loot$rust$cxxbridge1$File$condition(*this));
}

::rust::Str File::constraint() const noexcept {
  return ::rust::impl<::rust::Str>::new_unchecked(loot$rust$cxxbridge1$File$constraint(*this));
}

::rust::Box<::loot::rust::File> File::boxed_clone() const noexcept {
  return ::rust::Box<::loot::rust::File>::from_raw(loot$rust$cxxbridge1$File$boxed_clone(*this));
}

::std::size_t Filename::layout::size() noexcept {
  return loot$rust$cxxbridge1$Filename$operator$sizeof();
}

::std::size_t Filename::layout::align() noexcept {
  return loot$rust$cxxbridge1$Filename$operator$alignof();
}

::rust::Box<::loot::rust::Filename> new_filename(::rust::String name) noexcept {
  return ::rust::Box<::loot::rust::Filename>::from_raw(loot$rust$cxxbridge1$new_filename(&name));
}

::rust::Str Filename::as_str() const noexcept {
  return ::rust::impl<::rust::Str>::new_unchecked(loot$rust$cxxbridge1$Filename$as_str(*this));
}

::rust::Box<::loot::rust::Filename> Filename::boxed_clone() const noexcept {
  return ::rust::Box<::loot::rust::Filename>::from_raw(loot$rust$cxxbridge1$Filename$boxed_clone(*this));
}

bool Filename::eq(::loot::rust::Filename const &other) const noexcept {
  return loot$rust$cxxbridge1$Filename$eq(*this, other);
}

bool Filename::ne(::loot::rust::Filename const &other) const noexcept {
  return loot$rust$cxxbridge1$Filename$ne(*this, other);
}

bool Filename::lt(::loot::rust::Filename const &other) const noexcept {
  return loot$rust$cxxbridge1$Filename$lt(*this, other);
}

bool Filename::le(::loot::rust::Filename const &other) const noexcept {
  return loot$rust$cxxbridge1$Filename$le(*this, other);
}

bool Filename::gt(::loot::rust::Filename const &other) const noexcept {
  return loot$rust$cxxbridge1$Filename$gt(*this, other);
}

bool Filename::ge(::loot::rust::Filename const &other) const noexcept {
  return loot$rust$cxxbridge1$Filename$ge(*this, other);
}

::std::size_t Tag::layout::size() noexcept {
  return loot$rust$cxxbridge1$Tag$operator$sizeof();
}

::std::size_t Tag::layout::align() noexcept {
  return loot$rust$cxxbridge1$Tag$operator$alignof();
}

::rust::Box<::loot::rust::Tag> new_tag(::rust::String name, ::loot::rust::TagSuggestion suggestion, ::rust::Str condition) {
  ::rust::MaybeUninit<::rust::Box<::loot::rust::Tag>> return$;
  ::rust::repr::PtrLen error$ = loot$rust$cxxbridge1$new_tag(&name, suggestion, condition, &return$.value);
  if (error$.ptr) {
    throw ::rust::impl<::rust::Error>::error(error$);
  }
  return ::std::move(return$.value);
}

::rust::Str Tag::name() const noexcept {
  return ::rust::impl<::rust::Str>::new_unchecked(loot$rust$cxxbridge1$Tag$name(*this));
}

bool Tag::is_addition() const noexcept {
  return loot$rust$cxxbridge1$Tag$is_addition(*this);
}

::rust::Str Tag::condition() const noexcept {
  return ::rust::impl<::rust::Str>::new_unchecked(loot$rust$cxxbridge1$Tag$condition(*this));
}

::rust::Box<::loot::rust::Tag> Tag::boxed_clone() const noexcept {
  return ::rust::Box<::loot::rust::Tag>::from_raw(loot$rust$cxxbridge1$Tag$boxed_clone(*this));
}

::std::size_t PluginCleaningData::layout::size() noexcept {
  return loot$rust$cxxbridge1$PluginCleaningData$operator$sizeof();
}

::std::size_t PluginCleaningData::layout::align() noexcept {
  return loot$rust$cxxbridge1$PluginCleaningData$operator$alignof();
}

::rust::Box<::loot::rust::PluginCleaningData> new_plugin_cleaning_data(::std::uint32_t crc, ::rust::String cleaning_utility, ::rust::Slice<::rust::Box<::loot::rust::MessageContent> const> detail, ::std::uint32_t itm_count, ::std::uint32_t deleted_reference_count, ::std::uint32_t deleted_navmesh_count) {
  ::rust::MaybeUninit<::rust::Box<::loot::rust::PluginCleaningData>> return$;
  ::rust::repr::PtrLen error$ = loot$rust$cxxbridge1$new_plugin_cleaning_data(crc, &cleaning_utility, detail, itm_count, deleted_reference_count, deleted_navmesh_count, &return$.value);
  if (error$.ptr) {
    throw ::rust::impl<::rust::Error>::error(error$);
  }
  return ::std::move(return$.value);
}

::std::uint32_t PluginCleaningData::crc() const noexcept {
  return loot$rust$cxxbridge1$PluginCleaningData$crc(*this);
}

::std::uint32_t PluginCleaningData::itm_count() const noexcept {
  return loot$rust$cxxbridge1$PluginCleaningData$itm_count(*this);
}

::std::uint32_t PluginCleaningData::deleted_reference_count() const noexcept {
  return loot$rust$cxxbridge1$PluginCleaningData$deleted_reference_count(*this);
}

::std::uint32_t PluginCleaningData::deleted_navmesh_count() const noexcept {
  return loot$rust$cxxbridge1$PluginCleaningData$deleted_navmesh_count(*this);
}

::rust::Str PluginCleaningData::cleaning_utility() const noexcept {
  return ::rust::impl<::rust::Str>::new_unchecked(loot$rust$cxxbridge1$PluginCleaningData$cleaning_utility(*this));
}

::rust::Slice<::loot::rust::MessageContent const> PluginCleaningData::detail() const noexcept {
  return ::rust::impl<::rust::Slice<::loot::rust::MessageContent const>>::slice(loot$rust$cxxbridge1$PluginCleaningData$detail(*this));
}

::rust::Box<::loot::rust::PluginCleaningData> PluginCleaningData::boxed_clone() const noexcept {
  return ::rust::Box<::loot::rust::PluginCleaningData>::from_raw(loot$rust$cxxbridge1$PluginCleaningData$boxed_clone(*this));
}

::std::size_t Location::layout::size() noexcept {
  return loot$rust$cxxbridge1$Location$operator$sizeof();
}

::std::size_t Location::layout::align() noexcept {
  return loot$rust$cxxbridge1$Location$operator$alignof();
}

::rust::Box<::loot::rust::Location> new_location(::rust::String url, ::rust::Str name) noexcept {
  return ::rust::Box<::loot::rust::Location>::from_raw(loot$rust$cxxbridge1$new_location(&url, name));
}

::rust::Str Location::url() const noexcept {
  return ::rust::impl<::rust::Str>::new_unchecked(loot$rust$cxxbridge1$Location$url(*this));
}

::rust::Str Location::name() const noexcept {
  return ::rust::impl<::rust::Str>::new_unchecked(loot$rust$cxxbridge1$Location$name(*this));
}

::rust::Box<::loot::rust::Location> Location::boxed_clone() const noexcept {
  return ::rust::Box<::loot::rust::Location>::from_raw(loot$rust$cxxbridge1$Location$boxed_clone(*this));
}
} // namespace rust
} // namespace loot

extern "C" {
::loot::rust::Game *cxxbridge1$box$loot$rust$Game$alloc() noexcept;
void cxxbridge1$box$loot$rust$Game$dealloc(::loot::rust::Game *) noexcept;
void cxxbridge1$box$loot$rust$Game$drop(::rust::Box<::loot::rust::Game> *ptr) noexcept;

::loot::rust::Database *cxxbridge1$box$loot$rust$Database$alloc() noexcept;
void cxxbridge1$box$loot$rust$Database$dealloc(::loot::rust::Database *) noexcept;
void cxxbridge1$box$loot$rust$Database$drop(::rust::Box<::loot::rust::Database> *ptr) noexcept;

::loot::rust::OptionalPlugin *cxxbridge1$box$loot$rust$OptionalPlugin$alloc() noexcept;
void cxxbridge1$box$loot$rust$OptionalPlugin$dealloc(::loot::rust::OptionalPlugin *) noexcept;
void cxxbridge1$box$loot$rust$OptionalPlugin$drop(::rust::Box<::loot::rust::OptionalPlugin> *ptr) noexcept;

void cxxbridge1$rust_vec$loot$rust$Plugin$new(::rust::Vec<::loot::rust::Plugin> const *ptr) noexcept;
void cxxbridge1$rust_vec$loot$rust$Plugin$drop(::rust::Vec<::loot::rust::Plugin> *ptr) noexcept;
::std::size_t cxxbridge1$rust_vec$loot$rust$Plugin$len(::rust::Vec<::loot::rust::Plugin> const *ptr) noexcept;
::std::size_t cxxbridge1$rust_vec$loot$rust$Plugin$capacity(::rust::Vec<::loot::rust::Plugin> const *ptr) noexcept;
::loot::rust::Plugin const *cxxbridge1$rust_vec$loot$rust$Plugin$data(::rust::Vec<::loot::rust::Plugin> const *ptr) noexcept;
void cxxbridge1$rust_vec$loot$rust$Plugin$reserve_total(::rust::Vec<::loot::rust::Plugin> *ptr, ::std::size_t new_cap) noexcept;
void cxxbridge1$rust_vec$loot$rust$Plugin$set_len(::rust::Vec<::loot::rust::Plugin> *ptr, ::std::size_t len) noexcept;
void cxxbridge1$rust_vec$loot$rust$Plugin$truncate(::rust::Vec<::loot::rust::Plugin> *ptr, ::std::size_t len) noexcept;

void cxxbridge1$rust_vec$loot$rust$Message$new(::rust::Vec<::loot::rust::Message> const *ptr) noexcept;
void cxxbridge1$rust_vec$loot$rust$Message$drop(::rust::Vec<::loot::rust::Message> *ptr) noexcept;
::std::size_t cxxbridge1$rust_vec$loot$rust$Message$len(::rust::Vec<::loot::rust::Message> const *ptr) noexcept;
::std::size_t cxxbridge1$rust_vec$loot$rust$Message$capacity(::rust::Vec<::loot::rust::Message> const *ptr) noexcept;
::loot::rust::Message const *cxxbridge1$rust_vec$loot$rust$Message$data(::rust::Vec<::loot::rust::Message> const *ptr) noexcept;
void cxxbridge1$rust_vec$loot$rust$Message$reserve_total(::rust::Vec<::loot::rust::Message> *ptr, ::std::size_t new_cap) noexcept;
void cxxbridge1$rust_vec$loot$rust$Message$set_len(::rust::Vec<::loot::rust::Message> *ptr, ::std::size_t len) noexcept;
void cxxbridge1$rust_vec$loot$rust$Message$truncate(::rust::Vec<::loot::rust::Message> *ptr, ::std::size_t len) noexcept;

void cxxbridge1$rust_vec$loot$rust$Group$new(::rust::Vec<::loot::rust::Group> const *ptr) noexcept;
void cxxbridge1$rust_vec$loot$rust$Group$drop(::rust::Vec<::loot::rust::Group> *ptr) noexcept;
::std::size_t cxxbridge1$rust_vec$loot$rust$Group$len(::rust::Vec<::loot::rust::Group> const *ptr) noexcept;
::std::size_t cxxbridge1$rust_vec$loot$rust$Group$capacity(::rust::Vec<::loot::rust::Group> const *ptr) noexcept;
::loot::rust::Group const *cxxbridge1$rust_vec$loot$rust$Group$data(::rust::Vec<::loot::rust::Group> const *ptr) noexcept;
void cxxbridge1$rust_vec$loot$rust$Group$reserve_total(::rust::Vec<::loot::rust::Group> *ptr, ::std::size_t new_cap) noexcept;
void cxxbridge1$rust_vec$loot$rust$Group$set_len(::rust::Vec<::loot::rust::Group> *ptr, ::std::size_t len) noexcept;
void cxxbridge1$rust_vec$loot$rust$Group$truncate(::rust::Vec<::loot::rust::Group> *ptr, ::std::size_t len) noexcept;

::loot::rust::Group *cxxbridge1$box$loot$rust$Group$alloc() noexcept;
void cxxbridge1$box$loot$rust$Group$dealloc(::loot::rust::Group *) noexcept;
void cxxbridge1$box$loot$rust$Group$drop(::rust::Box<::loot::rust::Group> *ptr) noexcept;

void cxxbridge1$rust_vec$loot$rust$Vertex$new(::rust::Vec<::loot::rust::Vertex> const *ptr) noexcept;
void cxxbridge1$rust_vec$loot$rust$Vertex$drop(::rust::Vec<::loot::rust::Vertex> *ptr) noexcept;
::std::size_t cxxbridge1$rust_vec$loot$rust$Vertex$len(::rust::Vec<::loot::rust::Vertex> const *ptr) noexcept;
::std::size_t cxxbridge1$rust_vec$loot$rust$Vertex$capacity(::rust::Vec<::loot::rust::Vertex> const *ptr) noexcept;
::loot::rust::Vertex const *cxxbridge1$rust_vec$loot$rust$Vertex$data(::rust::Vec<::loot::rust::Vertex> const *ptr) noexcept;
void cxxbridge1$rust_vec$loot$rust$Vertex$reserve_total(::rust::Vec<::loot::rust::Vertex> *ptr, ::std::size_t new_cap) noexcept;
void cxxbridge1$rust_vec$loot$rust$Vertex$set_len(::rust::Vec<::loot::rust::Vertex> *ptr, ::std::size_t len) noexcept;
void cxxbridge1$rust_vec$loot$rust$Vertex$truncate(::rust::Vec<::loot::rust::Vertex> *ptr, ::std::size_t len) noexcept;

::loot::rust::OptionalPluginMetadata *cxxbridge1$box$loot$rust$OptionalPluginMetadata$alloc() noexcept;
void cxxbridge1$box$loot$rust$OptionalPluginMetadata$dealloc(::loot::rust::OptionalPluginMetadata *) noexcept;
void cxxbridge1$box$loot$rust$OptionalPluginMetadata$drop(::rust::Box<::loot::rust::OptionalPluginMetadata> *ptr) noexcept;

::loot::rust::PluginMetadata *cxxbridge1$box$loot$rust$PluginMetadata$alloc() noexcept;
void cxxbridge1$box$loot$rust$PluginMetadata$dealloc(::loot::rust::PluginMetadata *) noexcept;
void cxxbridge1$box$loot$rust$PluginMetadata$drop(::rust::Box<::loot::rust::PluginMetadata> *ptr) noexcept;

::loot::rust::Message *cxxbridge1$box$loot$rust$Message$alloc() noexcept;
void cxxbridge1$box$loot$rust$Message$dealloc(::loot::rust::Message *) noexcept;
void cxxbridge1$box$loot$rust$Message$drop(::rust::Box<::loot::rust::Message> *ptr) noexcept;

::loot::rust::MessageContent *cxxbridge1$box$loot$rust$MessageContent$alloc() noexcept;
void cxxbridge1$box$loot$rust$MessageContent$dealloc(::loot::rust::MessageContent *) noexcept;
void cxxbridge1$box$loot$rust$MessageContent$drop(::rust::Box<::loot::rust::MessageContent> *ptr) noexcept;

::loot::rust::OptionalCrc *cxxbridge1$box$loot$rust$OptionalCrc$alloc() noexcept;
void cxxbridge1$box$loot$rust$OptionalCrc$dealloc(::loot::rust::OptionalCrc *) noexcept;
void cxxbridge1$box$loot$rust$OptionalCrc$drop(::rust::Box<::loot::rust::OptionalCrc> *ptr) noexcept;

::loot::rust::Plugin *cxxbridge1$box$loot$rust$Plugin$alloc() noexcept;
void cxxbridge1$box$loot$rust$Plugin$dealloc(::loot::rust::Plugin *) noexcept;
void cxxbridge1$box$loot$rust$Plugin$drop(::rust::Box<::loot::rust::Plugin> *ptr) noexcept;

::loot::rust::Vertex *cxxbridge1$box$loot$rust$Vertex$alloc() noexcept;
void cxxbridge1$box$loot$rust$Vertex$dealloc(::loot::rust::Vertex *) noexcept;
void cxxbridge1$box$loot$rust$Vertex$drop(::rust::Box<::loot::rust::Vertex> *ptr) noexcept;

::loot::rust::File *cxxbridge1$box$loot$rust$File$alloc() noexcept;
void cxxbridge1$box$loot$rust$File$dealloc(::loot::rust::File *) noexcept;
void cxxbridge1$box$loot$rust$File$drop(::rust::Box<::loot::rust::File> *ptr) noexcept;

::loot::rust::Tag *cxxbridge1$box$loot$rust$Tag$alloc() noexcept;
void cxxbridge1$box$loot$rust$Tag$dealloc(::loot::rust::Tag *) noexcept;
void cxxbridge1$box$loot$rust$Tag$drop(::rust::Box<::loot::rust::Tag> *ptr) noexcept;

::loot::rust::PluginCleaningData *cxxbridge1$box$loot$rust$PluginCleaningData$alloc() noexcept;
void cxxbridge1$box$loot$rust$PluginCleaningData$dealloc(::loot::rust::PluginCleaningData *) noexcept;
void cxxbridge1$box$loot$rust$PluginCleaningData$drop(::rust::Box<::loot::rust::PluginCleaningData> *ptr) noexcept;

::loot::rust::Location *cxxbridge1$box$loot$rust$Location$alloc() noexcept;
void cxxbridge1$box$loot$rust$Location$dealloc(::loot::rust::Location *) noexcept;
void cxxbridge1$box$loot$rust$Location$drop(::rust::Box<::loot::rust::Location> *ptr) noexcept;

::loot::rust::Filename *cxxbridge1$box$loot$rust$Filename$alloc() noexcept;
void cxxbridge1$box$loot$rust$Filename$dealloc(::loot::rust::Filename *) noexcept;
void cxxbridge1$box$loot$rust$Filename$drop(::rust::Box<::loot::rust::Filename> *ptr) noexcept;
} // extern "C"

namespace rust {
inline namespace cxxbridge1 {
template <>
::loot::rust::Game *Box<::loot::rust::Game>::allocation::alloc() noexcept {
  return cxxbridge1$box$loot$rust$Game$alloc();
}
template <>
void Box<::loot::rust::Game>::allocation::dealloc(::loot::rust::Game *ptr) noexcept {
  cxxbridge1$box$loot$rust$Game$dealloc(ptr);
}
template <>
void Box<::loot::rust::Game>::drop() noexcept {
  cxxbridge1$box$loot$rust$Game$drop(this);
}
template <>
::loot::rust::Database *Box<::loot::rust::Database>::allocation::alloc() noexcept {
  return cxxbridge1$box$loot$rust$Database$alloc();
}
template <>
void Box<::loot::rust::Database>::allocation::dealloc(::loot::rust::Database *ptr) noexcept {
  cxxbridge1$box$loot$rust$Database$dealloc(ptr);
}
template <>
void Box<::loot::rust::Database>::drop() noexcept {
  cxxbridge1$box$loot$rust$Database$drop(this);
}
template <>
::loot::rust::OptionalPlugin *Box<::loot::rust::OptionalPlugin>::allocation::alloc() noexcept {
  return cxxbridge1$box$loot$rust$OptionalPlugin$alloc();
}
template <>
void Box<::loot::rust::OptionalPlugin>::allocation::dealloc(::loot::rust::OptionalPlugin *ptr) noexcept {
  cxxbridge1$box$loot$rust$OptionalPlugin$dealloc(ptr);
}
template <>
void Box<::loot::rust::OptionalPlugin>::drop() noexcept {
  cxxbridge1$box$loot$rust$OptionalPlugin$drop(this);
}
template <>
Vec<::loot::rust::Plugin>::Vec() noexcept {
  cxxbridge1$rust_vec$loot$rust$Plugin$new(this);
}
template <>
void Vec<::loot::rust::Plugin>::drop() noexcept {
  return cxxbridge1$rust_vec$loot$rust$Plugin$drop(this);
}
template <>
::std::size_t Vec<::loot::rust::Plugin>::size() const noexcept {
  return cxxbridge1$rust_vec$loot$rust$Plugin$len(this);
}
template <>
::std::size_t Vec<::loot::rust::Plugin>::capacity() const noexcept {
  return cxxbridge1$rust_vec$loot$rust$Plugin$capacity(this);
}
template <>
::loot::rust::Plugin const *Vec<::loot::rust::Plugin>::data() const noexcept {
  return cxxbridge1$rust_vec$loot$rust$Plugin$data(this);
}
template <>
void Vec<::loot::rust::Plugin>::reserve_total(::std::size_t new_cap) noexcept {
  return cxxbridge1$rust_vec$loot$rust$Plugin$reserve_total(this, new_cap);
}
template <>
void Vec<::loot::rust::Plugin>::set_len(::std::size_t len) noexcept {
  return cxxbridge1$rust_vec$loot$rust$Plugin$set_len(this, len);
}
template <>
void Vec<::loot::rust::Plugin>::truncate(::std::size_t len) {
  return cxxbridge1$rust_vec$loot$rust$Plugin$truncate(this, len);
}
template <>
Vec<::loot::rust::Message>::Vec() noexcept {
  cxxbridge1$rust_vec$loot$rust$Message$new(this);
}
template <>
void Vec<::loot::rust::Message>::drop() noexcept {
  return cxxbridge1$rust_vec$loot$rust$Message$drop(this);
}
template <>
::std::size_t Vec<::loot::rust::Message>::size() const noexcept {
  return cxxbridge1$rust_vec$loot$rust$Message$len(this);
}
template <>
::std::size_t Vec<::loot::rust::Message>::capacity() const noexcept {
  return cxxbridge1$rust_vec$loot$rust$Message$capacity(this);
}
template <>
::loot::rust::Message const *Vec<::loot::rust::Message>::data() const noexcept {
  return cxxbridge1$rust_vec$loot$rust$Message$data(this);
}
template <>
void Vec<::loot::rust::Message>::reserve_total(::std::size_t new_cap) noexcept {
  return cxxbridge1$rust_vec$loot$rust$Message$reserve_total(this, new_cap);
}
template <>
void Vec<::loot::rust::Message>::set_len(::std::size_t len) noexcept {
  return cxxbridge1$rust_vec$loot$rust$Message$set_len(this, len);
}
template <>
void Vec<::loot::rust::Message>::truncate(::std::size_t len) {
  return cxxbridge1$rust_vec$loot$rust$Message$truncate(this, len);
}
template <>
Vec<::loot::rust::Group>::Vec() noexcept {
  cxxbridge1$rust_vec$loot$rust$Group$new(this);
}
template <>
void Vec<::loot::rust::Group>::drop() noexcept {
  return cxxbridge1$rust_vec$loot$rust$Group$drop(this);
}
template <>
::std::size_t Vec<::loot::rust::Group>::size() const noexcept {
  return cxxbridge1$rust_vec$loot$rust$Group$len(this);
}
template <>
::std::size_t Vec<::loot::rust::Group>::capacity() const noexcept {
  return cxxbridge1$rust_vec$loot$rust$Group$capacity(this);
}
template <>
::loot::rust::Group const *Vec<::loot::rust::Group>::data() const noexcept {
  return cxxbridge1$rust_vec$loot$rust$Group$data(this);
}
template <>
void Vec<::loot::rust::Group>::reserve_total(::std::size_t new_cap) noexcept {
  return cxxbridge1$rust_vec$loot$rust$Group$reserve_total(this, new_cap);
}
template <>
void Vec<::loot::rust::Group>::set_len(::std::size_t len) noexcept {
  return cxxbridge1$rust_vec$loot$rust$Group$set_len(this, len);
}
template <>
void Vec<::loot::rust::Group>::truncate(::std::size_t len) {
  return cxxbridge1$rust_vec$loot$rust$Group$truncate(this, len);
}
template <>
::loot::rust::Group *Box<::loot::rust::Group>::allocation::alloc() noexcept {
  return cxxbridge1$box$loot$rust$Group$alloc();
}
template <>
void Box<::loot::rust::Group>::allocation::dealloc(::loot::rust::Group *ptr) noexcept {
  cxxbridge1$box$loot$rust$Group$dealloc(ptr);
}
template <>
void Box<::loot::rust::Group>::drop() noexcept {
  cxxbridge1$box$loot$rust$Group$drop(this);
}
template <>
Vec<::loot::rust::Vertex>::Vec() noexcept {
  cxxbridge1$rust_vec$loot$rust$Vertex$new(this);
}
template <>
void Vec<::loot::rust::Vertex>::drop() noexcept {
  return cxxbridge1$rust_vec$loot$rust$Vertex$drop(this);
}
template <>
::std::size_t Vec<::loot::rust::Vertex>::size() const noexcept {
  return cxxbridge1$rust_vec$loot$rust$Vertex$len(this);
}
template <>
::std::size_t Vec<::loot::rust::Vertex>::capacity() const noexcept {
  return cxxbridge1$rust_vec$loot$rust$Vertex$capacity(this);
}
template <>
::loot::rust::Vertex const *Vec<::loot::rust::Vertex>::data() const noexcept {
  return cxxbridge1$rust_vec$loot$rust$Vertex$data(this);
}
template <>
void Vec<::loot::rust::Vertex>::reserve_total(::std::size_t new_cap) noexcept {
  return cxxbridge1$rust_vec$loot$rust$Vertex$reserve_total(this, new_cap);
}
template <>
void Vec<::loot::rust::Vertex>::set_len(::std::size_t len) noexcept {
  return cxxbridge1$rust_vec$loot$rust$Vertex$set_len(this, len);
}
template <>
void Vec<::loot::rust::Vertex>::truncate(::std::size_t len) {
  return cxxbridge1$rust_vec$loot$rust$Vertex$truncate(this, len);
}
template <>
::loot::rust::OptionalPluginMetadata *Box<::loot::rust::OptionalPluginMetadata>::allocation::alloc() noexcept {
  return cxxbridge1$box$loot$rust$OptionalPluginMetadata$alloc();
}
template <>
void Box<::loot::rust::OptionalPluginMetadata>::allocation::dealloc(::loot::rust::OptionalPluginMetadata *ptr) noexcept {
  cxxbridge1$box$loot$rust$OptionalPluginMetadata$dealloc(ptr);
}
template <>
void Box<::loot::rust::OptionalPluginMetadata>::drop() noexcept {
  cxxbridge1$box$loot$rust$OptionalPluginMetadata$drop(this);
}
template <>
::loot::rust::PluginMetadata *Box<::loot::rust::PluginMetadata>::allocation::alloc() noexcept {
  return cxxbridge1$box$loot$rust$PluginMetadata$alloc();
}
template <>
void Box<::loot::rust::PluginMetadata>::allocation::dealloc(::loot::rust::PluginMetadata *ptr) noexcept {
  cxxbridge1$box$loot$rust$PluginMetadata$dealloc(ptr);
}
template <>
void Box<::loot::rust::PluginMetadata>::drop() noexcept {
  cxxbridge1$box$loot$rust$PluginMetadata$drop(this);
}
template <>
::loot::rust::Message *Box<::loot::rust::Message>::allocation::alloc() noexcept {
  return cxxbridge1$box$loot$rust$Message$alloc();
}
template <>
void Box<::loot::rust::Message>::allocation::dealloc(::loot::rust::Message *ptr) noexcept {
  cxxbridge1$box$loot$rust$Message$dealloc(ptr);
}
template <>
void Box<::loot::rust::Message>::drop() noexcept {
  cxxbridge1$box$loot$rust$Message$drop(this);
}
template <>
::loot::rust::MessageContent *Box<::loot::rust::MessageContent>::allocation::alloc() noexcept {
  return cxxbridge1$box$loot$rust$MessageContent$alloc();
}
template <>
void Box<::loot::rust::MessageContent>::allocation::dealloc(::loot::rust::MessageContent *ptr) noexcept {
  cxxbridge1$box$loot$rust$MessageContent$dealloc(ptr);
}
template <>
void Box<::loot::rust::MessageContent>::drop() noexcept {
  cxxbridge1$box$loot$rust$MessageContent$drop(this);
}
template <>
::loot::rust::OptionalCrc *Box<::loot::rust::OptionalCrc>::allocation::alloc() noexcept {
  return cxxbridge1$box$loot$rust$OptionalCrc$alloc();
}
template <>
void Box<::loot::rust::OptionalCrc>::allocation::dealloc(::loot::rust::OptionalCrc *ptr) noexcept {
  cxxbridge1$box$loot$rust$OptionalCrc$dealloc(ptr);
}
template <>
void Box<::loot::rust::OptionalCrc>::drop() noexcept {
  cxxbridge1$box$loot$rust$OptionalCrc$drop(this);
}
template <>
::loot::rust::Plugin *Box<::loot::rust::Plugin>::allocation::alloc() noexcept {
  return cxxbridge1$box$loot$rust$Plugin$alloc();
}
template <>
void Box<::loot::rust::Plugin>::allocation::dealloc(::loot::rust::Plugin *ptr) noexcept {
  cxxbridge1$box$loot$rust$Plugin$dealloc(ptr);
}
template <>
void Box<::loot::rust::Plugin>::drop() noexcept {
  cxxbridge1$box$loot$rust$Plugin$drop(this);
}
template <>
::loot::rust::Vertex *Box<::loot::rust::Vertex>::allocation::alloc() noexcept {
  return cxxbridge1$box$loot$rust$Vertex$alloc();
}
template <>
void Box<::loot::rust::Vertex>::allocation::dealloc(::loot::rust::Vertex *ptr) noexcept {
  cxxbridge1$box$loot$rust$Vertex$dealloc(ptr);
}
template <>
void Box<::loot::rust::Vertex>::drop() noexcept {
  cxxbridge1$box$loot$rust$Vertex$drop(this);
}
template <>
::loot::rust::File *Box<::loot::rust::File>::allocation::alloc() noexcept {
  return cxxbridge1$box$loot$rust$File$alloc();
}
template <>
void Box<::loot::rust::File>::allocation::dealloc(::loot::rust::File *ptr) noexcept {
  cxxbridge1$box$loot$rust$File$dealloc(ptr);
}
template <>
void Box<::loot::rust::File>::drop() noexcept {
  cxxbridge1$box$loot$rust$File$drop(this);
}
template <>
::loot::rust::Tag *Box<::loot::rust::Tag>::allocation::alloc() noexcept {
  return cxxbridge1$box$loot$rust$Tag$alloc();
}
template <>
void Box<::loot::rust::Tag>::allocation::dealloc(::loot::rust::Tag *ptr) noexcept {
  cxxbridge1$box$loot$rust$Tag$dealloc(ptr);
}
template <>
void Box<::loot::rust::Tag>::drop() noexcept {
  cxxbridge1$box$loot$rust$Tag$drop(this);
}
template <>
::loot::rust::PluginCleaningData *Box<::loot::rust::PluginCleaningData>::allocation::alloc() noexcept {
  return cxxbridge1$box$loot$rust$PluginCleaningData$alloc();
}
template <>
void Box<::loot::rust::PluginCleaningData>::allocation::dealloc(::loot::rust::PluginCleaningData *ptr) noexcept {
  cxxbridge1$box$loot$rust$PluginCleaningData$dealloc(ptr);
}
template <>
void Box<::loot::rust::PluginCleaningData>::drop() noexcept {
  cxxbridge1$box$loot$rust$PluginCleaningData$drop(this);
}
template <>
::loot::rust::Location *Box<::loot::rust::Location>::allocation::alloc() noexcept {
  return cxxbridge1$box$loot$rust$Location$alloc();
}
template <>
void Box<::loot::rust::Location>::allocation::dealloc(::loot::rust::Location *ptr) noexcept {
  cxxbridge1$box$loot$rust$Location$dealloc(ptr);
}
template <>
void Box<::loot::rust::Location>::drop() noexcept {
  cxxbridge1$box$loot$rust$Location$drop(this);
}
template <>
::loot::rust::Filename *Box<::loot::rust::Filename>::allocation::alloc() noexcept {
  return cxxbridge1$box$loot$rust$Filename$alloc();
}
template <>
void Box<::loot::rust::Filename>::allocation::dealloc(::loot::rust::Filename *ptr) noexcept {
  cxxbridge1$box$loot$rust$Filename$dealloc(ptr);
}
template <>
void Box<::loot::rust::Filename>::drop() noexcept {
  cxxbridge1$box$loot$rust$Filename$drop(this);
}
} // namespace cxxbridge1
} // namespace rust
